/*
This is an implementation of the AES algorithm, specifically ECB, CTR and CBC mode.
Block size can be chosen in aes.h - available choices are AES128, AES192, AES256.
The implementation is verified against the test vectors in:
  National Institute of Standards and Technology Special Publication 800-38A 2001 ED
ECB-AES128
----------
  plain-text:
    6bc1bee22e409f96e93d7e117393172a
    ae2d8a571e03ac9c9eb76fac45af8e51
    30c81c46a35ce411e5fbc1191a0a52ef
    f69f2445df4f9b17ad2b417be66c3710
  key:
    2b7e151628aed2a6abf7158809cf4f3c
  resulting cipher
    3ad77bb40d7a3660a89ecaf32466ef97 
    f5d3d58503b9699de785895a96fdbaaf 
    43b1cd7f598ece23881b00e3ed030688 
    7b0c785e27e8ad3f8223207104725dd4 
NOTE:   String length must be evenly divisible by 16byte (str_len % 16 == 0)
        You should pad the end of the string with zeros if this is not the case.
        For AES192/256 the key size is proportionally larger.
*/

#include <stdint.h>
#include <string.h> // CBC mode, for memset
#include "aes128cbc.h"

#define NB 4  // The number of columns comprising a state in AES. This is a constant in AES. Value=4
#define NK 4  // The number of 32 bit words in a key.
#define NR 10 // The number of rounds in AES Cipher.

/*
union state_u
{
    uint8_t d1[16];
    uint8_t d2[4][4];
};
typedef union state_u state_t; // state - array holding the intermediate results during decryption.
*/
static const uint8_t sbox[256] = {
  //0     1    2      3     4    5     6     7      8    9     A      B    C     D     E     F
  0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
  0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
  0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
  0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
  0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
  0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
  0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
  0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
  0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
  0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
  0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
  0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
  0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
  0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
  0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
  0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16 };

static const uint8_t rsbox[256] = {
  0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb,
  0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb,
  0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
  0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25,
  0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92,
  0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
  0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06,
  0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b,
  0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
  0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e,
  0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b,
  0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
  0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f,
  0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef,
  0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
  0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d };

// The round constant word array, Rcon[i], contains the values given by 
// x to the power (i-1) being powers of x (x is denoted as {02}) in the field GF(2^8)
static const uint8_t Rcon[11] = { 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36 };

// This function produces Nb(Nr+1) round keys. The round keys are used in each round to decrypt the states. 
static void keyExpansion(uint8_t* roundKey, const uint8_t* key)
{
  unsigned i, j, k;
  uint8_t tempa[4]; // Used for the column/row operations
  
  // The first round key is the key itself.
  for (i = 0; i < NK; ++i)
  {
    roundKey[i * 4 + 0] = key[i * 4 + 0];
    roundKey[i * 4 + 1] = key[i * 4 + 1];
    roundKey[i * 4 + 2] = key[i * 4 + 2];
    roundKey[i * 4 + 3] = key[i * 4 + 3];
  }

  // All other round keys are found from the previous round keys.
  for (i = NK; i < NB * (NR + 1); ++i)
  {
    {
      k = (i - 1) * 4;
      tempa[0] = roundKey[k + 0];
      tempa[1] = roundKey[k + 1];
      tempa[2] = roundKey[k + 2];
      tempa[3] = roundKey[k + 3];

    }

    if (i % NK == 0)
    {
      // This function shifts the 4 bytes in a word to the left once.
      // [a0,a1,a2,a3] becomes [a1,a2,a3,a0]

      // Function RotWord()
      {
        const uint8_t u8tmp = tempa[0];
        tempa[0] = tempa[1];
        tempa[1] = tempa[2];
        tempa[2] = tempa[3];
        tempa[3] = u8tmp;
      }

      // SubWord() is a function that takes a four-byte input word and 
      // applies the S-box to each of the four bytes to produce an output word.

      // Function Subword()
      {
        tempa[0] = sbox[tempa[0]];
        tempa[1] = sbox[tempa[1]];
        tempa[2] = sbox[tempa[2]];
        tempa[3] = sbox[tempa[3]];
      }

      tempa[0] = tempa[0] ^ Rcon[i/NK];
    }
    j = i * 4; k=(i - NK) * 4;
    roundKey[j + 0] = roundKey[k + 0] ^ tempa[0];
    roundKey[j + 1] = roundKey[k + 1] ^ tempa[1];
    roundKey[j + 2] = roundKey[k + 2] ^ tempa[2];
    roundKey[j + 3] = roundKey[k + 3] ^ tempa[3];
  }
}

// This function adds the round key to state.
// The round key is added to the state by an XOR function.
static void addRoundKey(uint8_t round, uint8_t* state, const uint8_t* roundKey)
{
    uint8_t i,j;
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            *(state + i*4 + j) ^= roundKey[(round * NB * 4) + (i * NB) + j];
        }
    }
}

// The SubBytes Function Substitutes the values in the
// state matrix with values in an S-box.
static void subBytes(uint8_t* state)
{
    uint8_t i, j;
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            *(state + j*4 + i) = sbox[*(state + j*4 + i)];
        }
    }
}

// The ShiftRows() function shifts the rows in the state to the left.
// Each row is shifted with different offset.
// Offset = Row number. So the first row is not shifted.
static void shiftRows(uint8_t* state)
{
    uint8_t temp;
    
    // Rotate first row 1 columns to left  
    temp             = *(state + 4*0 + 1);
    *(state + 4*0 + 1) = *(state + 4*1 + 1);
    *(state + 4*1 + 1) = *(state + 4*2 + 1);
    *(state + 4*2 + 1) = *(state + 4*3 + 1);
    *(state + 4*3 + 1) =   temp;
    
    // Rotate second row 2 columns to left  
    temp             = *(state + 4*0 + 2);
    *(state + 4*0 + 2) = *(state + 4*2 + 2);
    *(state + 4*2 + 2) =   temp;
    
    temp             = *(state + 4*1 + 2);
    *(state + 4*1 + 2) = *(state + 4*3 + 2);
    *(state + 4*3 + 2) =   temp;
    
    // Rotate third row 3 columns to left
    temp             = *(state + 4*0 + 3);
    *(state + 4*0 + 3) = *(state + 4*3 + 3);
    *(state + 4*3 + 3) = *(state + 4*2 + 3);
    *(state + 4*2 + 3) = *(state + 4*1 + 3);
    *(state + 4*1 + 3) =   temp;
}

static uint8_t xtime(uint8_t x)
{
    return ((x<<1) ^ (((x>>7) & 1) * 0x1b));
}

// MixColumns function mixes the columns of the state matrix
static void mixColumns(uint8_t* state)
{
    uint8_t i;
    uint8_t tmp, tm, t;
    for (i = 0; i < 4; ++i)
    {  
        t   = *(state + 4*i + 0);
        tmp = *(state + 4*i + 0) ^ *(state + 4*i + 1) ^ *(state + 4*i + 2) ^ *(state + 4*i + 3);
        tm  = *(state + 4*i + 0) ^ *(state + 4*i + 1) ; tm = xtime(tm);  *(state + 4*i + 0) ^= tm ^ tmp ;
        tm  = *(state + 4*i + 1) ^ *(state + 4*i + 2) ; tm = xtime(tm);  *(state + 4*i + 1) ^= tm ^ tmp ;
        tm  = *(state + 4*i + 2) ^ *(state + 4*i + 3) ; tm = xtime(tm);  *(state + 4*i + 2) ^= tm ^ tmp ;
        tm  = *(state + 4*i + 3) ^ t ;                  tm = xtime(tm);  *(state + 4*i + 3) ^= tm ^ tmp ;
    }
}

static uint8_t multiply(uint8_t x, uint8_t y)
{
    return (((y>>0 & 1) * x) ^
            ((y>>1 & 1) * xtime(x)) ^
            ((y>>2 & 1) * xtime(xtime(x))) ^
            ((y>>3 & 1) * xtime(xtime(xtime(x)))) ^
            ((y>>4 & 1) * xtime(xtime(xtime(xtime(x)))))); /* this last call to xtime() can be omitted */
}

// MixColumns function mixes the columns of the state matrix.
// The method used to multiply may be difficult to understand for the inexperienced.
// Please use the references to gain more information.
static void invMixColumns(uint8_t* state)
{
    int i;
    uint8_t a, b, c, d;
    for (i = 0; i < 4; ++i)
    { 
        a = *(state + 4*i + 0);
        b = *(state + 4*i + 1);
        c = *(state + 4*i + 2);
        d = *(state + 4*i + 3);

        *(state + 4*i + 0) = multiply(a, 0x0e) ^ multiply(b, 0x0b) ^ multiply(c, 0x0d) ^ multiply(d, 0x09);
        *(state + 4*i + 1) = multiply(a, 0x09) ^ multiply(b, 0x0e) ^ multiply(c, 0x0b) ^ multiply(d, 0x0d);
        *(state + 4*i + 2) = multiply(a, 0x0d) ^ multiply(b, 0x09) ^ multiply(c, 0x0e) ^ multiply(d, 0x0b);
        *(state + 4*i + 3) = multiply(a, 0x0b) ^ multiply(b, 0x0d) ^ multiply(c, 0x09) ^ multiply(d, 0x0e);
    }
}


// The SubBytes Function Substitutes the values in the
// state matrix with values in an S-box.
static void invSubBytes(uint8_t* state)
{
    uint8_t i, j;
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            *(state + 4*j + i) = rsbox[*(state + 4*j + i)];
        }
    }
}

static void invShiftRows(uint8_t* state)
{
    uint8_t temp;
    
    // Rotate first row 1 columns to right  
    temp             = *(state + 4*3 + 1);
    *(state + 4*3 + 1) = *(state + 4*2 + 1);
    *(state + 4*2 + 1) = *(state + 4*1 + 1);
    *(state + 4*1 + 1) = *(state + 4*0 + 1);
    *(state + 4*0 + 1) =   temp;
    
    // Rotate second row 2 columns to right 
    temp             = *(state + 4*0 + 2);
    *(state + 4*0 + 2) = *(state + 4*2 + 2);
    *(state + 4*2 + 2) =   temp;
    
    temp             = *(state + 4*1 + 2);
    *(state + 4*1 + 2) = *(state + 4*3 + 2);
    *(state + 4*3 + 2) =   temp;
    
    // Rotate third row 3 columns to right
    temp             = *(state + 4*0 + 3);
    *(state + 4*0 + 3) = *(state + 4*1 + 3);
    *(state + 4*1 + 3) = *(state + 4*2 + 3);
    *(state + 4*2 + 3) = *(state + 4*3 + 3);
    *(state + 4*3 + 3) =   temp;
}

// Cipher is the main function that encrypts the PlainText.
static void cipher(uint8_t* state, const uint8_t* roundKey)
{
    uint8_t round = 0;
    
    // Add the First round key to the state before starting the rounds.
    addRoundKey(0, state, roundKey); 
    
    // There will be Nr rounds.
    // The first Nr-1 rounds are identical.
    // These Nr-1 rounds are executed in the loop below.
    for (round = 1; round < NR; ++round)
    {
        subBytes(state);
        shiftRows(state);
        mixColumns(state);
        addRoundKey(round, state, roundKey);
    }
    
    // The last round is given below.
    // The MixColumns function is not here in the last round.
    subBytes(state);
    shiftRows(state);
    addRoundKey(NR, state, roundKey);
}

static void invCipher(uint8_t* state, const uint8_t* roundKey)
{
    uint8_t round = 0;
    
    // Add the First round key to the state before starting the rounds.
    addRoundKey(NR, state, roundKey); 
    
    // There will be Nr rounds.
    // The first Nr-1 rounds are identical.
    // These Nr-1 rounds are executed in the loop below.
    for (round = (NR - 1); round > 0; --round)
    {
        invShiftRows(state);
        invSubBytes(state);
        addRoundKey(round, state, roundKey);
        invMixColumns(state);
    }
    
    // The last round is given below.
    // The MixColumns function is not here in the last round.
    invShiftRows(state);
    invSubBytes(state);
    addRoundKey(0, state, roundKey);
}

/*****************************************************************************/
/* Public functions:                                                         */
/*****************************************************************************/

void Aes128CbcEncrypt(const uint8_t* key, const uint8_t* iv, uint8_t* buf, uint32_t length)
{
    uint8_t ctxRoundKey[176];
    keyExpansion(ctxRoundKey, key);
    
    const uint8_t* prevIv = iv;
    for (int i = 0; i < length; i += AES128CBC_BLOCK_SIZE)
    {
        for (int j = 0; j < AES128CBC_BLOCK_SIZE; ++j) buf[j] ^= prevIv[j];
        cipher(buf, ctxRoundKey);
        prevIv = buf;
        buf += AES128CBC_BLOCK_SIZE;
    }
}
void Aes128CbcDecrypt(const uint8_t* key, const uint8_t* iv, uint8_t* buf, uint32_t length)
{
    uint8_t ctxRoundKey[176];
    keyExpansion(ctxRoundKey, key);

    uint8_t ctxIv[AES128CBC_BLOCK_SIZE];
    memcpy (ctxIv, iv, AES128CBC_BLOCK_SIZE);
    
    uint8_t storeNextIv[AES128CBC_BLOCK_SIZE];
    for (int i = 0; i < length; i += AES128CBC_BLOCK_SIZE)
    {
        memcpy(storeNextIv, buf, AES128CBC_BLOCK_SIZE);
        invCipher(buf, ctxRoundKey);
        for (int j = 0; j < AES128CBC_BLOCK_SIZE; ++j) buf[j] ^= ctxIv[j];
        memcpy(ctxIv, storeNextIv, AES128CBC_BLOCK_SIZE);
        buf += AES128CBC_BLOCK_SIZE;
    }
}
