#include <stdint.h>
#include <stdbool.h>
#include <arm_compat.h> //Used for the __[en/dis]able_irq intrinsics

#include "rtc.h"
#include "tm.h"
#include "clk.h"
#include "clktime.h"
#include "clkgov.h"
#include "hrtimer.h"
#include "led.h"
#include "log.h"

static clktime tickCount = 0;
static clktime slewCount = 0;
static bool   countIsSet = false;

bool    ClkTimeIsSet() { return countIsSet; }

static uint32_t hrTimerAtLastIncrement = 0; //Set by the increment function

clktime ClkTimeGet()
{
    return tickCount + slewCount + HrTimerProRata(CLK_TIME_ONE_SECOND + ClkGovGetPpb() + ClkGovGetSlew(), HrTimerSince(hrTimerAtLastIncrement));
}

void ClkTimeSet(clktime extClock)
{    
     clktime timerCountSinceLastSecond = HrTimerSince(hrTimerAtLastIncrement);
     clktime fraction = (timerCountSinceLastSecond << CLK_TIME_ONE_SECOND_SHIFT) / HR_TIMER_COUNT_PER_SECOND;
     clktime    ticks = extClock - fraction;

    __disable_irq();
        tickCount = ticks;
        slewCount = 0;
    __enable_irq();
    
    countIsSet = true;
}
void ClkTimeAdjustSeconds(int seconds)
{
    __disable_irq();
        tickCount += (clktime)seconds << CLK_TIME_ONE_SECOND_SHIFT;
    __enable_irq();
}
void ClkTimeIncrementByOneSecond(uint32_t startCount)
{
    __disable_irq();
        hrTimerAtLastIncrement = startCount;
        tickCount += CLK_TIME_ONE_SECOND + ClkGovGetPpb();
        slewCount += ClkGovGetSlew();
        ClkGovSetSlew(0);
    __enable_irq();
}

static volatile  clktime  tickSnapshot;
static volatile  clktime  slewSnapshot;
static volatile uint32_t timerSnapshot;

void ClkTimeSaveSnapshot()
{
     timerSnapshot = HrTimerSince(hrTimerAtLastIncrement);
      tickSnapshot = tickCount;
      slewSnapshot = slewCount;
}
void ClkTimesGetFromSnapshot(clktime* pInt, clktime* pAbs)
{
    *pInt = tickSnapshot                + HrTimerProRata(CLK_TIME_ONE_SECOND + ClkGovGetPpb(),                   timerSnapshot);
    *pAbs = tickSnapshot + slewSnapshot + HrTimerProRata(CLK_TIME_ONE_SECOND + ClkGovGetPpb() + ClkGovGetSlew(), timerSnapshot);
}