/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"
#include "ble/services/BatteryService.h"
#include "ble/services/DeviceInformationService.h"

#include "toolchain.h"
#include "TMP_nrf51/TMP_nrf51.h"
#include "TemperatureGatewayService.h"
#include "TemperatureTable.h"

#define MAX_BEACONS 10

BLE  ble;
DigitalOut led1(LED1);

TemperatureGatewayService<MAX_BEACONS>* tempGatewayServicePtr;
TemperatureTable<uint32_t, float, MAX_BEACONS> temperatureTable;

/* Setup device name and new service information */
const static char     DEVICE_NAME[]        = "TGW1";
static const uint16_t uuid16_list[]        = {GattService::UUID_HEALTH_THERMOMETER_SERVICE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE};

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    ble.gap().startAdvertising(); // restart advertising
}

void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */
}

void advertisementCallback(const Gap::AdvertisementCallbackParams_t *params) {

    struct AdvertisingData_t {
        uint8_t                        length; /* doesn't include itself */
        GapAdvertisingData::DataType_t dataType;
        uint8_t                        data[0];
    } PACKED;

    struct ApplicationData_t {
        uint16_t applicationSpecificId;             /* An ID used to identify temperature value
                                                       in the manufacture specific AD data field */
        TMP_nrf51::TempSensorValue_t tmpSensorValue; /* User defined application data */
    } PACKED;

    /* This ID is not very meaningful, but makes it easy to find the device for debugging */
    static const uint16_t APP_SPECIFIC_ID_TEST = 0xFEFE;

    /* Search for the manufacturer specific data with matching application-ID */
    AdvertisingData_t *pAdvData;
    for (size_t index = 0; index < params->advertisingDataLen; index += (pAdvData->length + 1)) {
        pAdvData = (AdvertisingData_t *)&params->advertisingData[index];

        if (pAdvData->dataType != GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA) {
            continue;
        }

        const ApplicationData_t *pAppData = (const ApplicationData_t *)pAdvData->data;
        if (pAppData->applicationSpecificId == APP_SPECIFIC_ID_TEST) {
            uint32_t beaconId = (params->peerAddr[3] << 24) | (params->peerAddr[2] << 16) | (params->peerAddr[1] << 8)  | (params->peerAddr[0]);
            TMP_nrf51::TempSensorValue_t temperature = (TMP_nrf51::TempSensorValue_t)pAppData->tmpSensorValue;

            /* If new data was added to the table, then update the payload */
            temperatureTable.addBeacon(beaconId, temperature);
            if (temperatureTable.hasUpdate()) {
                tempGatewayServicePtr->updateTemperature(temperatureTable.getTotalBeacons(), temperatureTable.getBeaconIds(), temperatureTable.getTemperatures());
                temperatureTable.resetHasUpdate();
            }
            break;
        }
    }
}

void configureScanTemperature(void)
{
    ble.gap().setScanParams(500 /* scan interval */, 200 /* scan window */);
    ble.gap().startScan(advertisementCallback);
}

int main(void)
{
    /* Blink LED every second to tell whether the app is running */
    led1 = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, 1);

    ble.init();

    configureScanTemperature();

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service. */
    TemperatureGatewayService<MAX_BEACONS> temperatureGatewayService(ble);
    tempGatewayServicePtr = &temperatureGatewayService;

    /* Setup auxiliary service. */
    DeviceInformationService deviceInfo(ble, "ARM", "Model1", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");

    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms */
    ble.gap().startAdvertising();

    /* infinite loop */
    while (1) {
        ble.waitForEvent(); // low power wait for event
    }
}
