#include "serial_bus.h"
#include "mbed.h"
#include "rtos.h"
#include "Queue/queue.h"

/* CAN BUS */
CAN HSCAN(p30, p29);

void can_bus_rx_IRQ()
{
    uint8 msg_raw[12];
    CANMessage msg;
    if(HSCAN.read(msg))
    {
       msg_raw[0] = (uint8)(msg.id >> 24);
       msg_raw[1] = (uint8)(msg.id >> 16);
       msg_raw[2] = (uint8)(msg.id >> 8);
       msg_raw[3] = (uint8)(msg.id >> 0);
       for(int i=0;i<8;i++)
        msg_raw[i+4] = msg.data[i];
        
       send_serial_bus(msg_raw,12);
    }
}

void can_init(void)
{
    HSCAN.frequency(500000);
    HSCAN.attach(can_bus_rx_IRQ,CAN::RxIrq);
}

void can_send(uint32 id, uint8 * data)
{
    HSCAN.write(CANMessage((int)id, (char *)data, 8));
}

/***********************/

extern Serial pc;
RawSerial uart0(p13,p14);


#define PACKETS       10
#define PACKET_SIZE   518
#define BAUDRATE    921600

Queue uartQueue(1,PACKETS * PACKET_SIZE);

/********************************/

void serial_data_callback()
{
    while(uart0.readable()) 
    {
       uint8 data = uart0.getc();
       uartQueue.PutIrq(&data);
    }
    
}

#define STARTOFFRAME    4
#define HEADER          0
#define PAYLOAD         1
#define CHECKSUM        2


uint16 success = 0;
uint16 errors = 0;

void wait_packet_layer(void)
{
    static uint8 packet[PACKET_SIZE];
    uint16 packet_index = 0;
    uint8 packet_state = STARTOFFRAME;
    uint16 packet_size = 0;
    uint8 packet_checksum = 0;
    
    /* flush the data */
    while(uart0.readable()) uart0.getc();
    
    uart0.attach(&serial_data_callback, Serial::RxIrq);
    wait(1.0);
   
    uint8 data;
    while(1)
    {
       
       /* get queue data */
       while(uartQueue.Get(&data))
       {
           switch(packet_state)
            {
                case STARTOFFRAME:
                    if(data == 0xC0)
                    {
                        packet_state = HEADER;
                        packet_index = 0;
                    }
                    break;
                case HEADER:
                    packet[packet_index++] = data;
                if(packet_index == 4)
                {
                    /* validating the packet header */
                    if( (packet[0] ^ packet[2]) == 0xFF && (packet[1] ^ packet[3]) == 0xFF )
                     {
                        packet_size = ((uint16)packet[0] << 8) | (packet[1]);
                        packet_state = PAYLOAD;
                        packet_index = 0;
                        packet_checksum = 0;
                     }
                     else
                     {
                       pc.printf("Wrong header\r\n");
                       packet_state = STARTOFFRAME;
                       packet_index = 0;  
                       errors++;                     
                     }
                }
                break;
                
                case PAYLOAD:
                    packet[packet_index++] = data;
                    packet_checksum = packet_checksum ^ data;
                    if(packet_index == (packet_size + 1))
                    {
                        if(packet_checksum == 0)
                        {
                            packet_state = STARTOFFRAME;
                            packet_index = 0;
                            /* mark the packet for processing */
                            //pc.printf("Success = %d, Errors = %d \r\n",success,errors);
                            //send_serial_bus(packet,packet_size);
                            can_send(0x7E0,packet);
                            success++;
                        }
                        else
                        {
                            pc.printf("Wrong checksum\r\n");
                            /* dropping the packet */
                            packet_state = STARTOFFRAME;
                            packet_index = 0;
                            errors++;
                        }                        
                    }
                break;
              }
          }
    }
}


void send_serial_bus(uint8 * packet, uint16 size)
{
    uint8 cks = 0;
    uart0.putc(0xC0);
    uart0.putc((size >> 8) & 0xFF);
    uart0.putc((size >> 0) & 0xFF);
    size = size ^ 0xFFFF;
    uart0.putc((size >> 8) & 0xFF);
    uart0.putc((size >> 0) & 0xFF);
    size = size ^ 0xFFFF;
    
    while(size > 0)
    {
        if(uart0.writeable())
        {
            cks = cks ^ (*packet);
            uart0.putc(*packet++);    
            size--;  
        }
    }
    uart0.putc(cks);
    
        
}

Thread uartThread;

void start_serial_bus(void)
{
    can_init();
    uart0.baud(BAUDRATE);
    uart0.attach(&serial_data_callback,Serial::RxIrq);
    uartThread.start(&wait_packet_layer);
}
