#include "ringbuf.h"
#include "types.h"

uint8 ICACHE_FLASH_ATTR ringbuf_init(ringbuf_t *r, uint8* buf, uint16 size)
{
    if( (r == NULL) || (buf == NULL) || (size < 2) )
        return FALSE;
    
    r->p_o = r->p_r = r->p_w = buf;
    r->fill_cnt = 0;
    r->size = size;
    r->p_e = r->p_o + r->size;
    return TRUE;
}

uint8 ICACHE_FLASH_ATTR ringbuf_put(ringbuf_t *r, uint8 c)
{
    if(r->fill_cnt>=r->size)
        return FALSE;                       // ring buffer is full, this should be atomic operation

    r->fill_cnt++;                          // increase filled slots count, this should be atomic operation

    *r->p_w++ = c;                          // put character into buffer

    if(r->p_w >= r->p_e)                        // rollback if write pointer go pass
        r->p_w = r->p_o;                        // the physical boundary
    
    return TRUE;
}

uint8 ICACHE_FLASH_ATTR ringbuf_get(ringbuf_t *r, uint8 * c)
{
    if(r->fill_cnt == 0)
        return FALSE;               // ring buffer is empty, this should be atomic operation
    
    r->fill_cnt--;                              // decrease filled slots count
    
    *c = *r->p_r++;                             // get the character out
    
    if(r->p_r >= r->p_e)                        // rollback if write pointer go pass
        r->p_r = r->p_o;                        // the physical boundary
    
    return TRUE;
}

uint16 ICACHE_FLASH_ATTR ringbuf_gets(ringbuf_t *r, uint8 * buff, uint16 size)
{
    uint16 read=0;
    while(size)
    {
        if(ringbuf_get(r,buff++) == FALSE)
            break;

        read++;
        size--;
    }
    return read;
}

uint16 ICACHE_FLASH_ATTR ringbuf_puts(ringbuf_t *r, uint8 * buff, uint16 size)
{
    uint16 write=0;
    while(size)
    {
        if(ringbuf_put(r,*buff++) == FALSE)
            break;

        write++;
        size--;
    }
    return write;
}

uint8 ICACHE_FLASH_ATTR ringbuf_clear(ringbuf_t *r)
{
    r->p_r = r->p_w = r->p_o;
    r->fill_cnt = 0;
    return TRUE;
}