/* Copyright (c) <2018> <MultiTech Systems>, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ChannelPlans.h"
#include "ModeRegion.h"
#include "MTSLog.h"

ModeRegion::ModeRegion(
    DOGS102* lcd,
    ButtonHandler* buttons,
    mDot* dot,
    LoRaHandler* lora,
    GPSPARSER* gps,
    SensorHandler* sensors,
    const char* file_cp)
    : Mode(lcd, buttons, dot, lora, gps, sensors)
    , _file_cp(file_cp)
{
    // Add all of our supported regions
    _regions.push_back("US915");
    _regions.push_back("AU915");
    _regions.push_back("EU868");
    _regions.push_back("AS923");
    _regions.push_back("AS923-JAPAN");
    _regions.push_back("KR920");
    _regions.push_back("IN865");

    _menu = new LayoutScrollSelect(lcd, _regions, "Select Region");
}

ModeRegion::~ModeRegion()
{
    delete _menu;
}

bool ModeRegion::start()
{
    std::string region("");
    
    // clear any stale signals
    osSignalClear(_main_id, buttonSignal | loraSignal);

    _menu->display();

    while (true) {
        osEvent e = Thread::signal_wait(0, 250);
        if (e.status == osEventSignal) {
            if (e.value.signals & buttonSignal) {
                _be = _buttons->getButtonEvent();

                switch (_be) {
                case ButtonHandler::sw1_press:
                    region = _menu->select();
                    setFreqBand(region);
                    return true;
                case ButtonHandler::sw2_press:
                    _menu->scroll();
                    break;
                case ButtonHandler::sw1_hold:
                    return true;
                default:
                    break;
                }
            }
        }
    }
}

void ModeRegion::setFreqBand(std::string region)
{
    uint8_t cp[] = { 0 };
    mDot::mdot_file file = _dot->openUserFile(_file_cp, mDot::FM_RDWR);

    // First delete the existing plan
    lora::ChannelPlan *plan = _dot->getChannelPlan();
    if (plan != NULL)
        delete plan;

    // Construct a channel plan for the selected region
    if (region == "US915") {
        plan = new lora::ChannelPlan_US915();
        cp[0] = CP_US915;
    } else if (region == "AU915") {
        plan = new lora::ChannelPlan_AU915();
        cp[0] = CP_AU915;
    } else if (region == "EU868") {
        plan = new lora::ChannelPlan_EU868();
        cp[0] = CP_EU868;
    } else if (region == "AS923") {
        plan = new lora::ChannelPlan_AS923();
        cp[0] = CP_AS923;
    } else if (region == "AS923-JAPAN") {
        plan = new lora::ChannelPlan_AS923_Japan();
        cp[0] = CP_AS923_JAPAN;
    } else if (region == "KR920") {
        plan = new lora::ChannelPlan_KR920();
        cp[0] = CP_KR920;
    } else if (region == "IN865") {
        plan = new lora::ChannelPlan_IN865();
        cp[0] = CP_IN865;
    }

    if (_dot->setChannelPlan(plan) == mDot::MDOT_OK) {
        // Update the channel plan config
        _dot->seekUserFile(file, 0, SEEK_SET);
        _dot->writeUserFile(file, cp, 1);
        _dot->closeUserFile(file);
        _band = _dot->getFrequencyBand();
        logInfo("Setting channel plan to %s", region.c_str());
    } else {
        logError("Failed to set channel plan to %s", region.c_str());
    }
}
