/**

@file Barometer.h
@brief Header file containing member functions and variables
@author Amir M. Asemanpayeh
@date May 2015

@brief Header file - Contains Global Variables, GPIO and Pins and Function prototypes

*/


#include "mbed.h"
#include "N5110.h"
#include "BMP180.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

/**
@namespace bmp180
@brief A I2C instance to interact with the BMP sensor \n SDA, SCL
*/
BMP180 bmp180(p28,p27);


/**
@namespace lcd
@brief A serial data protocol to operate with a master/slave relationship to interact with the lcd \n VCC,SCE,RST,D/C,MOSI,SCLK,LED
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);


/**
@namespace leds
@brief GPIO bus output for status LEDs
*/
BusOut leds(LED4,LED3,LED2,LED1);


/**
@namespace serial
@brief A USB Virtual Serial Port to print out messages to a host PC terminal
*/
Serial serial(USBTX,USBRX);


/**
@namespace local
@brief A local file system declared to store data files
*/
LocalFileSystem local("local");


/**
@namespace button1
@brief An Interrupt on digital pushbutton1 input p16
*/
InterruptIn button1(p16);


/**
@namespace button2
@brief An Interrupt on digital pushbutton2 input p17
*/
InterruptIn button2(p17);


/**
@namespace button3
@brief An Interrupt on digital pushbutton3 input p15
*/
InterruptIn button3 (p15);


/**
@namespace button4
@brief An Interrupt on digital pushbutton4 input p18
*/
InterruptIn button4 (p18);




/**
@namespace redLED
@brief A PwmOut interface is used to control the frequency and mark-space ratio of a led (The red LED)
*/
PwmOut redLED(p24);//red


/**
@namespace yellowLED
@brief A PwmOut interface is used to control the frequency and mark-space ratio of a led (The yellow LED)
*/
PwmOut yellowLED(p23);//yellow


/**
@namespace greenLED
@brief A PwmOut interface is used to control the frequency and mark-space ratio of a led (The green LED)
*/
PwmOut greenLED(p22);


/**
@namespace buzzer
@brief A PwmOut interface is used to control the frequency and mark-space ratio of a buzzer
pulse train
*/
PwmOut buzzer(p21);


/**
@namespace timer
@brief Create ticker object for temperature and pressure readings
*/
Ticker timer;


/**
@namespace dataLoggerTimer
@brief a ticker object for the data logger functions
*/
Ticker dataLoggerTimer ;


/**
@namespace powerSaverTimeout
@brief create a timeout object for the poweerSaver function
*/
Timeout powerSaverTimeout;

/**
@namespace debounce
@brief create a debounce timer object to implement a debounce counter to avoid multiple interrupts.
*/
Timer debounce;


int timerFlag = 1; /*!< Flag for tempesrature pressure reading ISR */
int tickerFlag = 0 ;  /*!<Flag to show the minutes passed */
int button1Flag = 0; /*!< Flag set in button1Pressed ISR */
int button2Flag = 0 ; /*!<Flag set in button2Pressed ISR  */
int button3Flag = 0 ; /*!< Flag set in button3Pressed ISR  */
int button4Flag = 0 ; /*!< Flag set in button4Pressed ISR */
int unitFlag = 1 ; /*!< Flag set in the unitSettings function, indicates the units picked by user */
int powerSaverFlag = 0 ; /*!<Flag set in the powerSettings function when the powerSaver is switched on */
int lcdPowerFlag = 1 ; /*!< Flag set in the savePower function to indicate the status of the LCD (ON or OFF) */
int dataLoggerFlag = 0 ; /*!< Flag set by user in the dataLoggerSetting function to turn on the logger */
int runLoggerFlag = 0 ;  /*!< Fag set when the logger is switched on, allows user to navigate to this option in the main menu */
int alarmClockFlag = 0 ; /*!< Flag set in the AlarmClock function, indicates if the alarm clock has been turned on */
int thresholdAlarmFlag = 0 ; /*!<Flag set in the thresholdAlarmmenu function,indicates if the thresholds are set */

int UNIXdate=0;   /*!<An integer to store the value of UNIX time calculated by mktime function */
int currentTime ;/*!<An integer to store the UNIX value of the current time */
int powerSaverTime = 60;    /*!<An integer to set the initial value of the power saver ticker */
int dataLoggerTime = 60;    /*!<An integer used to set the initial value of the data logger ticker */

float temperature;  /*!<A float value to store the value of the temperature recieved from the sensor */
float pressure; /*!<A float value to store the value of the pressure recieved from the sensor */
float averageTemperature ;  /*!<A float value to store the average value of temperature readingsn taken over a period of time */
float averagePressure ; /*!<A float value to store the average value of pressure readings over a period of time */
float minTemperature ;  /*!<A float value to save the minimum value of temperature readings */
float maxTemperature ;  /*!<A float value to store the maximum value of temperature readings */
float minPressure;  /*!<A float value to store the minimum value of pressure readings */
float maxPressure;  /*!<A float value to store the maximum value of pressure readings */




float arrayT[83] ; /*!<A A float array to store the values of temperature to calculate the average,minimum and maximum values*/
float arrayP[83] ; /*!<A A float array to store the values of pressure to calculate the average,minimum and maximum values*/
float arrayTG[83] ; /*!<A A float array to store the values of temperature to plot the related graph */
float arrayPG[83] ;/*!<A A float array to store the values of pressure to plot the related graph */


char bufferTime[14]; /*!<A A char array (buffer) to store time*/
char bufferDate[14]; /*!<A A char array (buffer) to store date*/
char bufferT[14];/*!<A A char array (buffer) to store the value of temperature in char*/
char bufferP[14];  /*!<A A char array (buffer) to store the value of pressure in char */
// each character is 6 pixels wide, screen is 84 pixels (84/6 = 14)
// so can display a string of a maximum 14 characters in length
// or create formatted strings - ensure they aren't more than 14 characters long



char buffer0[14];   /*!<A A char array (buffer) used to display data on the LCD at (0,0)*/
char buffer1[14];   /*!<A A char array (buffer) used to display data on the LCD at (0,1)*/
char buffer2[14];   /*!<A A char array (buffer) used to display data on the LCD at (0,2)*/
char buffer3[14];   /*!<A A char array (buffer) used to display data on the LCD at (0,3)*/
char buffer4[14];   /*!<A A char array (buffer) used to display data on the LCD at (0,4)*/
char buffer5[14];   /*!<A A char array (buffer) used to display data on the LCD at (0,5)*/



//arrays used to store data to the local file
char timeSent[1000]; /*!<A A char array used to save time staps to the local file */
char dateSent[1000]; /*!<A A char array used to save date stamps the local file */
char temperatureSent[1000]; /*!<A A char array used to save the temperature readings to the local file */
char pressureSent[1000]; /*!<A A char array used to save the pressure readings to the local file */
char temperatureRecieved[1000]; /*!<A A char array used to retrive the saved temperature readings from the local file */
char pressureRecieved[1000]; /*!<A A char array used to retrive the saved pressure readings from the local file  */
char timeRecieved[1000]; /*!<A A char array used to retrive the saved time stamps from the local file  */
char dateRecieved[1000]; /*!<A A char array used to retrive the saved date stamps from the local file */



char * pch; /*!<A A char array used to split the char arrays of temperature values retrived from the local file to the individual temperature strings */
char * pch1; /*!<A A char array used to split the char arrays of pressure values retrived from the local file to the individual pressure strings*/
char * pch2; /*!<A  A char array used to split the char arrays of times retrived from the local file to the individual time strings*/
char * pch3; /*!<A  A char array used to split the char arrays of dates retrived from the local file to the individual date strings*/


int state=0; /*!<A An integer used to navigate through the finite machines (indexes)*/

float frequency[]= {659,554,659,554,659,554,659,554,659,554,659,554}; /*!<A An array of floats with different frequencies for the buzzer*/



/**
A structure, contains the output, titles and nextstate data
@param output - integer value holds the output value of the members
@param title - char array contains titles of the members
@param nextState - int array contains the values for the next and previous states
*/
struct  State {
    int output;
    char title[14];
    int nextState[2];
};


/**
Initialises a typedef struct, contains the output, titles and nextstate data for the start menu
*/
typedef struct State STyp1; // for the start menu
STyp1 fsmA[4] = {
    {0,"Live Data",{1,3}},
    {1,"Saved Data",{2,0}},
    {2,"Alarms",{3,1}},
    {3,"Settings",{0,2}},
};


/**
Initialises a typedef struct, contains the output, titles and nextstate data for the settings menu
*/
typedef struct State STyp2; // for the settings menu
STyp2 fsmB[4] = {
    {0,"Date/Time",{1,3}},
    {1,"Units",{2,0}},
    {2,"Power Saver",{3,1}},
    {3,"Data Logger",{0,2}},
};

/**
Initialises a typedef struct, contains the output, titles and nextstate data for the time/date settings also the alarmclock function
*/
typedef struct State STyp3; // for the time/date settings
STyp3 fsmC[5] = {
    {12,"Hour:",{1,4}},
    {0,"Min:",{2,0}},
    {21,"Day:",{3,1}},
    {4,"Month:",{4,2}},
    {2015,"Year:",{0,3}},
};


/**
Initialises a typedef struct, contains the output, titles and nextstate data for the unitsettings function
*/
typedef struct State STyp4; // for the units settings
STyp3 fsmD[4] = {
    {1,"C/mb",{1,3}},
    {2,"C/atm",{2,0}},
    {3,"F/mb",{3,1}},
    {4,"F/atm",{0,2}},
};

/**
Initialises a typedef struct, contains the output, titles and nextstate data for the powerSaverSetting
*/
typedef struct State STyp5; // for the Power saver settings
STyp5 fsmE[2] = {
    {0,"Off",{1,1}},
    {60,"On",{0,0}},
};

/**
Initialises a typedef struct, contains the output, titles and nextstate data for the alarms menu
*/
typedef struct State Stype6; // for the alarms menu
Stype6 fsmF[2] = {
    {0,"Thresholds",{1,1}},
    {1,"Alarm Clock", {0,0}},
};

/**
Initialises a typedef struct, contains the output, titles and nextstate data for the threshold and alarmclock functions
*/
typedef struct State Stype7; // yes or no
Stype7 fsmG[2] = {
    {1,"Yes",{1,1}},
    {0,"No", {0,0}},
};

/**
Initialises a typedef struct, contains the output, titles and nextstate data for the threshold menu
*/
typedef struct State Stype8; // for the thresholds
Stype8 fsmH[4] = {
    {20,"Min T:",{1,3}},
    {30,"Max T:",{2,0}},
    {1000,"Min P:",{3,1}},
    {1100,"Max P:",{0,2}},
};


/**
ISR attached to the fisrt button interrupt
@param button1Flag - integer to indicate if the interrupt has occured
*/
void button1Pressed();

/**
ISR attached to the second button interrupt
@param button2Flag - integer to indicate if the interrupt has occured
*/
void button2Pressed();

/**
ISR attached to the third button interrupt
@param button3Flag - integer to indicate if the interrupt has occured
*/
void button3Pressed();
/**
ISR attached to the fourth button interrupt
@param button4Flag - integer to indicate if the interrupt has occured
*/
void button4Pressed();




/**
ISR attached to the ticker timer
@param timerFlag - integer to indicate the timer is expired
*/
void timerExpired();

/**
ISR attached to the ticker dataLoggerTimer
@param dataLoggerFlag - integer to indicate the data logger timer is expired
*/
void dataLoggerTimerExpired ();


/**
Opens the local folder, creates a csv file and stores the arrays.
@param data - char array inserted in the local file
@param data1 - char array inserted in the local file
@param data2 - char array inserted in the local file
@param data3 - char array inserted in the local file
*/
void saveToFile(char *data,char *data1,char *data2,char *data3);




/**
Gets the local time from the RTC and stores them into two arrays
@param currentTime - integer to store the current data epoch unix time stamp
@param bufferTime - char array to store the time stamp
@param bufferTime - char array to store the date stamp
*/
void updateTime();



/**
Displays the char arrays on the LCD
@param buffer 0 - char array printed on (0,0)
@param buffer 1 - char array printed on (0,1)
@param buffer 2 - char array printed on (0,2)
@param buffer 3 - char array printed on (0,3)
@param buffer 4 - char array printed on (0,4)
@param buffer 5 - char array printed on (0,5)
*/
void display();

/**
Reads the data from the sensor and stores them into arrays
@param bufferT - char array to store temperature
@param bufferP - char array to store pressure
@param temperature - float to store the value of the temperature
@param pressure - float to store the value of the pressure
*/
void readData();


/**
ISR attached to power saver timeout to set the flag to 1
@param powerSaverFlag - integer to indicate if the state of the power saver
*/
void powerSaverExpired();


/**
Checks if the power saver option is set by the user, if yes puts the device in sleep mode and turns off the lcd.
Also if the buttons are pressed it wakes up the device and attached the timeout
*/
void powerSaverCheck();



/**
Indicates the live data on the screen and updates the data once a second
@param timerFlag - integer set when the timer ends
*/
void liveData();

/**
Recieves the data, puts them in arrays and saves them to the file. Saved arrays are restored and used to calculate the minimun maximum average values
@param sumTemperature - integer to store the total value of temperature values
@param sumPressure - integer to store the total value of pressure values

@param temperatureSent - char with the temperature values saved to the file
@param pressureSent- char with the pressure values saved to the file
@param timeSent- char with the time stamps saved to the file
@param dateSent- char with the data stamps saved to the file

@param temperatureRecieved - char with the temperature values restored from the file
@param pressureRecieved- char with the pressure values restored from the file
@param timeRecieved- char with the time stamps restored from the file
@param dateRecieved- char with the data stamps restored from the file

@param minTemperature - int to store the minimum value of temperature
@param minPressure - int to store the minimum value of pressure
@param maxTemperature - int to store the maximum value of temperature
@param maxPressure - int to store the maximum value of pressure
@param avgTemperature - int to store the average value of temperature
@param avgPressure - int to store the average value of pressure
*/
void loggerData();


/**
Uses the data from the loggerData function to indicate the calculated values and plot the graphs
*/
void dataLogger();



/**
Checks the current values of temperature and pressure and compares to the threshold values, triggers the alarm if match
*/
void thresholdCheck();


/**
Alloes the user to set the values for the thresholds and also turn the alarm on or off
*/
void threshold();

/**
Compares the set time in the alarm clock section with the current time, triggers the alarm when they match
*/
void alarmClockCheck();



/**
Allows the user to set the min , hour, day, month and year values to set the alarm clock
*/
void alarmClock ();

/**
Alarm menu, includes two submenus : alarmclock and thresholds
*/
void alarmsMenu ();

/**
*Fourth option in the settings menu./n
*Allows the user to turn on the data logger and set the value of time out /n
*Attaches a timeout to set the dataloggerFlag /n
@param dataLoggerFlag - integer used as a flag to indicate if the datalogger option is switched on or off
@param state - integer used to navigate through fsm states
*/
void dataLoggerSetting();

/**
*Third option in the settings menu./n
*Allows the user to turn on the power saving option and set the value of time out /n
*Attaches a timeout to set the powerSaverFlag /n
@param powerSaverFlag - integer used as a flag to indicate if the power saver option is switched on or off
@param state - integer used to navigate through fsm states
*/
void powerSaverSetting();

/**
*Calculates the UNIX time value./n
*Sets the integer values entered by user to the tm structure mebers /n
*Converts the tm structure to time_t /n
@param timeinfo - tm struct sent to mktime function
@param rawtime - fundamental arithmetic type to represent the current time informations
*/
void calculateUNIXTime();

/**
*The second option in the settings menu./n
*Allows the user to change the units /n
@param unitFlag - integer used as a flag to indicate the units set
@param state - integer used to navigate through fsm states
*/
void unitsSetting();


/**
*The first option in the settings menu./n
*Allows the user to enter time and date /n
*Sends the recieved int arrays to the calculateUNIXtime function to calculate the unix time /n
*sets the current time to the calculated unix time.
@param state - integer used to navigate through fsm states
*/
void timeDateSetting();


/**
*The setting menu displayed when the fourth state of fsmA is picked from the main menu /n
*Depends on the state of the finite state machine used for this function, put the related title in the screen buffers /n
*Also copy the bufferTime (buffer used to store time) on the first screen buffer (at the top) /n
*Displays the screen buffers on the LCD/n
*switch statement checks the current state of the fsm (fsmB),
*The related function to each fsm state is executed as when a rising edge from the 4th button interrupts (button 4 is pressed)

*/
void settingsMenu();

/**
*The main menu displayed at the start /n
*Depends on the state of the finite state machine (fsmA) used for this function, put the related title in the screen buffers /n
*Also copy the bufferTime (buffer used to store time) on the first screen buffer (at the top) /n
*Displays the screen buffers on the LCD/n
*switch statement checks the current state of the fsm (fsmA),
*The related function to each fsm state is executed as when a rising edge from the 4th button interrupts (button 4 is pressed)

*/
void startMenu();





