/*
    The MIT License (MIT)
    
    Copyright (c) 2015 Tom Soulanille
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
    
    @file          udp.cpp 
    @purpose       simple networking
    @version       0.1
    @date          31 March 2015
    @author        Tom Soulanille    
*/

#include "snet.h"

#if 0
//-- from util.cpp
extern void print_hex(const uint8_t *p, int len);
#endif

void Snet::turn_udp_packet_around(uint8_t *buf) {
    uint8_t dh, dl;
    turn_ip_packet_around(buf);
    dh = buf[UDP_DST_PORT_O];
    dl = buf[UDP_DST_PORT_O+1];
    buf[UDP_DST_PORT_O] = buf[UDP_SRC_PORT_O];
    buf[UDP_DST_PORT_O+1] = buf[UDP_SRC_PORT_O+1];
    buf[UDP_SRC_PORT_O] = dh;
    buf[UDP_SRC_PORT_O+1] = dl;
}

// Send a UDP packet addressed as in buf, with specified payload
void Snet::send_udp_packet(uint8_t *buf, const uint8_t *payload, int payload_len) {
    int ip_total_length = UDP_PAYLOAD_O - IP_HEADER_O + payload_len;
    int udp_length = UDP_HEADER_LEN + payload_len;
    buf[IP_TOTAL_LENGTH_O] = ip_total_length >> 8;
    buf[IP_TOTAL_LENGTH_O+1] = ip_total_length;

    buf[UDP_LENGTH_O] = udp_length >> 8;
    buf[UDP_LENGTH_O+1] = udp_length;
#if 0
    // RFC1768: "An all zero  transmitted
    // checksum  value means that the transmitter  generated  no checksum  (for
    // debugging or for higher level protocols that don't care)."
    buf[UDP_CHECKSUM_O] = buf[UDP_CHECKSUM_O+1] = 0;
#else
// find the udp checksum for a given enet packet presumed to be udp
// From rfc768:
//  The pseudo  header  conceptually prefixed to the UDP header contains the
//  source  address,  the destination  address,  the protocol,  and the  UDP
//  length.   This information gives protection against misrouted datagrams.
//  This checksum procedure is the same as is used in TCP.
//  
//                    0      7 8     15 16    23 24    31
//                   +--------+--------+--------+--------+
//                   |          source address           |
//                   +--------+--------+--------+--------+
//                   |        destination address        |
//                   +--------+--------+--------+--------+
//                   |  zero  |protocol|   UDP length    |
//                   +--------+--------+--------+--------+
//  
//  If the computed  checksum  is zero,  it is transmitted  as all ones (the
//  equivalent  in one's complement  arithmetic).   An all zero  transmitted
//  checksum  value means that the transmitter  generated  no checksum  (for
//  debugging or for higher level protocols that don't care).
    
    // This is the pseudo-header sum
    int sum = ones_complement_sum(&buf[IP_SADDR_O], 4+4, 0);
    sum += 0x11;    // proto
    sum += 8 + payload_len; // UDP length
    
    // add the real UDP header, with zeroed-out checksum
    buf[UDP_CHECKSUM_O] = buf[UDP_CHECKSUM_O+1] = 0;
    sum = ones_complement_sum(&buf[UDP_HEADER_O],
            UDP_PAYLOAD_O - UDP_HEADER_O,
            sum);
            
    // add the payload that will be included and take the result
    int chksum = ~ones_complement_sum(payload, payload_len, sum);
    
    chksum = chksum ? chksum : ~chksum; // send -0 for +0
    buf[UDP_CHECKSUM_O] = chksum >> 8;
    buf[UDP_CHECKSUM_O+1] = chksum;
#endif
    buf[IP_CHKSUM_O] = buf[IP_CHKSUM_O+1] = 0;
    int chk = ip_checksum_of(&buf[IP_HEADER_O], IP_HEADER_LEN);
    buf[IP_CHKSUM_O] = chk >> 8;
    buf[IP_CHKSUM_O+1] = chk;
    eth.write((char *) buf, UDP_PAYLOAD_O);
    eth.write((char *) payload, payload_len);
    eth.send();
#if 0
    printf("rup wrote:");
    print_hex(buf, UDP_PAYLOAD_O);
    printf("then wrote:");
    print_hex((uint8_t *) payload, payload_len);
    printf("udp chksum 0x%02x%02x\r\n", buf[UDP_CHECKSUM_O], buf[UDP_CHECKSUM_O+1]);
#endif
}

// Return a UDP packet to the sender of packet in buf, with new payload
void Snet::return_udp_packet(uint8_t *buf, const uint8_t *payload, int payload_len) {
    turn_udp_packet_around(buf);
    send_udp_packet(buf, payload, payload_len);
}

void Snet::send_to_correspondent(const uint8_t *what, int what_len) {
    send_udp_packet(correspondent_facing_packet_header, what, what_len);
}

void Snet::tell_udp_correspondent(char *s) {
    send_to_correspondent((const uint8_t *) s, strlen(s));
}

void Snet::set_udp_correspondent(uint8_t *buf, int len) {
    turn_udp_packet_around(buf);
    memcpy(correspondent_facing_packet_header, buf, UDP_PAYLOAD_O);
}

void Snet::handle_udp_packet(uint8_t *buf, int len) {
#if 0
    printf("UDP:");
    print_hex(&buf[UDP_HEADER_O], len-UDP_HEADER_O);
#endif
    if (registered_udp_handler) (*registered_udp_handler)(buf, len);
};
