/*
    The MIT License (MIT)
    
    Copyright (c) 2015 Tom Soulanille
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
    
    @file          inet.cpp
    @purpose       simple networking
    @version       0.1
    @date          31 March 2015
    @author        Tom Soulanille    
*/

#include "snet.h"

void handle_icmp_packet(uint8_t *buf, int len);
void handle_udp_packet(uint8_t *buf, int len);

//extern Ethernet eth;
//extern uint8_t my_mac[];
//uint8_t my_ip[4] = { 192, 168, 32, 191 };

//extern uint16_t ip_checksum_of(const uint8_t *buf, int len);
//extern uint16_t ones_complement_sum(const uint8_t *buf, const int len, int sum);

//-- IP utilities

// Calculate the ones' complement sum of a buffer, initialized
// See also RFC1071 http://tools.ietf.org/html/rfc1071
uint16_t Snet::ones_complement_sum(const uint8_t *buf, int len, int sum) {
    for (int i=0; i<len/2; i++) {
        sum += (buf[0] << 8) + (buf[1]);
        buf += 2;
    }
    if (len & 1) sum += buf[0] << 8;    // odd length
    while (sum & 0xffff0000) sum = (sum & 0xffff) + (sum >> 16);
    return sum;
}

// Calculate the RFC1071 IP checksum. No assumption about byte alignment is made.
uint16_t Snet::ip_checksum_of(const uint8_t *buf, const int len) {
    return ~ones_complement_sum(buf, len, 0);
};

void Snet::turn_ip_packet_around(uint8_t *buf) {
    memcpy(&buf[ENET_DMAC_O], &buf[ENET_SMAC_O], 6);
    memcpy(&buf[ENET_SMAC_O], my_mac, 6);
    memcpy(&buf[IP_DADDR_O], &buf[IP_SADDR_O], 4);
    memcpy(&buf[IP_SADDR_O], my_ip, 4);
};


void Snet::handle_icmp_packet(uint8_t *buf, int len) {
#if 0
    printf("ICMP:");
    print_hex(&buf[ICMP_HEADER_O], len-ICMP_HEADER_O);
#endif
#if 0
    printf("ip header checksum = 0x%x\r\n",
            ip_checksum_of(&buf[IP_HEADER_O], IP_HEADER_LEN));
#endif
#if 0
    printf("icmp checksum = 0x%x\r\n",
            ip_checksum_of(&buf[ICMP_HEADER_O], len-ICMP_HEADER_O));
#endif
    if (buf[ICMP_TYPE_O] != 8 || buf[ICMP_CODE_O] != 0) return;
    // Here we have ICMP echo request
#if 0
    printf("ICMP echo request\r\n");
#endif
    buf[ICMP_TYPE_O] = 0x0; // make it an echo reply
    // Fix the ICMP checksum
    // HC' = ~(C + (-m) + m')    --    [Eqn. 3]
    //     = ~(~HC + ~m + m')
    // HC' = HC - ~m - m'    --    [Eqn. 4] of RFC1624, where:
    //    HC  - old checksum in header
    //    C   - one's complement sum of old header
    //    HC' - new checksum in header
    //    m   - old value of a 16-bit field
    //    m'  - new value of a 16-bit field
    // we have m == 0x800 and m' == 0
    // so HC' = HC - ~0x800 = HC - 0xf7ff
    uint32_t sum = (buf[ICMP_CHECKSUM_O] << 8)
                    + buf[ICMP_CHECKSUM_O + 1]
                    - 0xf7ff;
    if (sum & 0xffff0000) sum -= 1; // handle borrow-around
    //printf("adj: 0x%x\r\n", sum);
    buf[ICMP_CHECKSUM_O] = sum >> 8;    // type casting takes care of the masking
    buf[ICMP_CHECKSUM_O + 1] = sum;
    turn_ip_packet_around(buf);
    eth.write((char *) buf, len);
    eth.send();
#if 0
    printf("check new checksums: ip = 0x%x, icmp = 0x%x\r\n",
            ip_checksum_of(&buf[IP_HEADER_O], IP_HEADER_LEN),
            ip_checksum_of(&buf[ICMP_HEADER_O], len-ICMP_HEADER_O));
#endif

#if 1
    // Test we got the checksums right
    if (int v = ip_checksum_of(&buf[IP_HEADER_O], IP_HEADER_LEN))
            printf("Bad new IP checksum: %d\r\n", v);
    if (int v = ip_checksum_of(&buf[ICMP_HEADER_O], len-ICMP_HEADER_O))
            printf("Bad new ICMP checksum: %d\r\n", v);
#endif

#if 0
    printf("sent");
    print_hex(buf, len);
#endif
};


void Snet::interpret_inet_packet(uint8_t *buf, int len) {
    //printf("Got inet:");
    //print_hex(&buf[ENET_PAYLOAD_O], len-IP_HEADER_O);    //DEBUG
    // skip packets with certain attributes we don't like
#if 0
    printf("ip header checksum = 0x%x\r\n",
            ip_checksum_of(&buf[IP_HEADER_O], IP_HEADER_LEN));
#endif
    if (buf[IP_VIHL_O] != 0x45
            || buf[IP_FLAGS_FRAGOFF_O+0] & ~0x40 != 0x00    // allow Don't Fragment flag
            || buf[IP_FLAGS_FRAGOFF_O+1] != 0x00
            || memcmp(&buf[IP_DADDR_O], my_ip, 4)) {
        printf("don't like: fragoff 0x%x, dest %d.%d.%d.%d\r\n",
                (buf[IP_FLAGS_FRAGOFF_O+0] << 8) + buf[IP_FLAGS_FRAGOFF_O+1],
                buf[IP_DADDR_O+0],
                buf[IP_DADDR_O+1],
                buf[IP_DADDR_O+2],
                buf[IP_DADDR_O+3]
                );   //DEBUG
        return;
    };
    if (buf[IP_PROTO_O] == 0x01) handle_icmp_packet(buf, len);
    else if (buf[IP_PROTO_O] == 0x11) handle_udp_packet(buf, len);
#if 0
    else printf("Protocol 0x%x\r\n", buf[IP_PROTO_O]);
#endif
}
