#include "mbed.h"
#include "mbed_debug.h"
#include "rtos.h"
#include "rtc.h"
#include "sd.h"
#include "SDFileSystem.h"
#include "ConfigFile.h"

Mutex sd_mutex;
SDFileSystem sd(MBED_SPI0, "sd");
ConfigFile config;
ConfigFile sensor_cfg;

bool read_config()
{
    bool b = true;
    sd_mutex.lock();
    if (b && !config.read("/sd/emir2.cfg")) b = false;
    if (b && !sensor_cfg.read("/sd/sensors.cfg")) b = false;
    sd_mutex.unlock();
    return b;
}


bool write_config()
{
    bool b = true;
    sd_mutex.lock();
    if (b && !config.write("/sd/emir2.cfg", "# EMIR2 main configuration", ConfigFile::DOS)) b = false;
    if (b && !sensor_cfg.write("/sd/sensors.cfg", "# EMIR2 sensor configuration", ConfigFile::DOS)) b = false;
    sd_mutex.unlock();
    return b;
}


int get_config(char *key, int def)
{
    char buffer[32];
    
    if (!config.getValue(key, buffer, sizeof(buffer))) return def;
    return atoi(buffer);
}


bool set_config(char *key, int value)
{
    char buffer[32];
    
    sprintf(buffer, "%d", value);
    if (!config.setValue(key, buffer)) return false;
    return true;    
}


int get_sensor_cfg(uint8_t *romid, char *desc)
{
    char romid_str[20];
    char buffer[32];
    char *pch;
    int index;
    
    for (int i = 0; i < 8; i++) sprintf(&romid_str[i*2], "%.2X", romid[i]);
    if (!sensor_cfg.getValue(romid_str, buffer, sizeof(buffer))) {
        sensor_cfg.setValue(romid_str, "0;Unknown");
        return 0;
    }
    
    pch = strchr(buffer, ';');
    if (pch == NULL) return 0;
    
    *pch = 0;
    index = atoi(buffer);
    
    pch++;
    if (desc != NULL) strcpy(desc, pch);
    
    return index;
}


bool log_temperature(int sensor_id, int value)
{
    char buffer[32];
    sprintf(buffer, "%d", value);
    return log_event(101, sensor_id, buffer);
}


bool log_io(int io_id, int value)
{
    char buffer[32];
    sprintf(buffer, "%d", value);
    return log_event(102, io_id, buffer);
}


bool log_event(int event_id, int sub_id, char *value)
{
    sd_mutex.lock();
    FILE *fp = fopen("/sd/emir2.log", "a");
    if (fp == NULL) {
        debug("Could not open file for write\n");
        sd_mutex.unlock();
        return false;
    }

    time_t m_time = get_rtc();
    struct tm *now;
    now = localtime(&m_time);

    fprintf(fp, "%d.%d.%04d %02d:%02d:%02d;%lu;", 
        now->tm_mday, now->tm_mon+1, now->tm_year+1900,
        now->tm_hour, now->tm_min, now->tm_sec, 
        m_time
    );
    fprintf(fp, "%d;%d;%s\r\n", event_id, sub_id, value);
    
    fclose(fp);
    sd_mutex.unlock();
    return true;
}


void sd_init()
{
    debug("Reading configs... ");
    bool b = read_config();
    if (b) debug("OK\n"); else debug("failed\n");

    debug("Writing log... ");
    log_event(100, 0, "EMIRv2 compiled " __DATE__ " " __TIME__ ", starting");
    debug("done\n");
}

