#ifndef _FONT_H_
#define _FONT_H_

/*
According to limitation of EPD resolution and customer requirements,
  only 12x12 used in EPD as standard Chinese font as Middle-standard size,
  its Latin Character uses 6x12 as standard font size,
  so, small, middle(standard), large and extra large, which is
  6x8, 6x12, 8x16, 12x24

EPD Scan parameters:
  Font Scan direction: Vertical, 
  Font Scan order: MSB first,
  Byte order: Column by Column

Other options are:
  4x6/5x7/6x8/8x12/8x16(ASC16)/12x16/12x24(ASC24)/14x24

Reference URLs:
  http://www.henningkarlsen.com/electronics
  http://bbs.ednchina.com/BLOG_ARTICLE_61130.HTM ==>ASCII Font ROM from ASC16
  http://blog.163.com/zsf_04/blog/static/786932092010112632658815/ ==>ASCII Font ROM

Font ZIMO Abstractor:
  PCtoLCD2002.exe, freeware for any font in any size
  UCDOS7.0 for all ASC* and HZK* bitmap font files
*/

#include "mbed.h"

/* Font from PC2LCD2002, colum scan, MSB first, Fixed System font */

/*
 !"#$%&'
()*+,-./
01234567
89:;<=>?
@ABCDEFG
HIJKLMNO
PQRSTUVW
XYZ[\]^_
`abcdefg
hijklmno
pqrstuvw
zyx{|}~
*/

/*
 !"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_`abcdefghijklmnopqrstuvwxyz{|}~
*/

/* ROM Font Index
  (0) !(1) "(2) #(3) $(4) %(5) &(6) '(7)
 ((8) )(9) *(10) +(11) ,(12) -(13) .(14) /(15)
 0(16) 1(17) 2(18) 3(19) 4(20) 5(21) 6(22) 7(23)
 8(24) 9(25) :(26) ;(27) <(28) =(29) >(30) ?(31)
 @(32) A(33) B(34) C(35) D(36) E(37) F(38) G(39)
 H(40) I(41) J(42) K(43) L(44) M(45) N(46) O(47)
 P(48) Q(49) R(50) S(51) T(52) U(53) V(54) W(55)
 X(56) Y(57) Z(58) [(59) \(60) ](61) ^(62) _(63)
 `(64) a(65) b(66) c(67) d(68) e(69) f(70) g(71)
 h(72) i(73) j(74) k(75) l(76) m(77) n(78) o(79)
 p(80) q(81) r(82) s(83) t(84) u(85) v(86) w(87)
 x(88) y(89) z(90) {(91) |(92) }(93) ~(94)
*/

const uint8_t EPD_S_Font0608[] = 
{  
0x04,0x08,0x20,0x5F, //12x08 Terminal
0x00,0x00,0x00,0x00,0x00,0x00,/*" ",0*/
0x00,0x00,0xFA,0x00,0x00,0x00,/*"!",1*/
0x00,0xC0,0x00,0xC0,0x00,0x00,/*""",2*/
0x28,0xFE,0x28,0xFE,0x28,0x00,/*"#",3*/
0x00,0x32,0x5E,0xF2,0x4C,0x00,/*"$",4*/
0x00,0x66,0x68,0x16,0x66,0x00,/*"%",5*/
0x00,0x1C,0xF2,0xB2,0x4C,0x00,/*"&",6*/
0x00,0x00,0x00,0xC0,0x00,0x00,/*"'",7*/
0x00,0x00,0x7C,0x82,0x00,0x00,/*"(",8*/
0x00,0x00,0x82,0x7C,0x00,0x00,/*")",9*/
0x00,0x38,0x7C,0x38,0x00,0x00,/*"*",10*/
0x00,0x10,0x7C,0x10,0x00,0x00,/*"+",11*/
0x00,0x00,0x00,0x06,0x00,0x00,/*",",12*/
0x00,0x10,0x10,0x10,0x10,0x00,/*"-",13*/
0x00,0x00,0x02,0x00,0x00,0x00,/*".",14*/
0x00,0x02,0x0C,0x30,0xC0,0x00,/*"/",15*/
0x00,0x7C,0x82,0x82,0x7C,0x00,/*"0",16*/
0x00,0x42,0xFE,0x02,0x00,0x00,/*"1",17*/
0x00,0x46,0x8A,0x92,0x62,0x00,/*"2",18*/
0x00,0x44,0x92,0x92,0x6C,0x00,/*"3",19*/
0x00,0x1C,0x64,0xFE,0x04,0x00,/*"4",20*/
0x00,0xF2,0x92,0x92,0x8C,0x00,/*"5",21*/
0x00,0x7C,0x92,0x92,0x4C,0x00,/*"6",22*/
0x00,0xC0,0x8E,0x90,0xE0,0x00,/*"7",23*/
0x00,0x6C,0x92,0x92,0x6C,0x00,/*"8",24*/
0x00,0x64,0x92,0x92,0x7C,0x00,/*"9",25*/
0x00,0x00,0x24,0x00,0x00,0x00,/*":",26*/
0x00,0x02,0x24,0x00,0x00,0x00,/*";",27*/
0x00,0x10,0x28,0x44,0x82,0x00,/*"<",28*/
0x00,0x28,0x28,0x28,0x28,0x00,/*"=",29*/
0x00,0x82,0x44,0x28,0x10,0x00,/*">",30*/
0x00,0x40,0x8A,0x90,0x60,0x00,/*"?",31*/
0x00,0x70,0x8E,0x92,0x7E,0x00,/*"@",32*/
0x00,0x7E,0x88,0x88,0x7E,0x00,/*"A",33*/
0x00,0xFE,0x92,0x92,0x6C,0x00,/*"B",34*/
0x00,0x7C,0x82,0x82,0x44,0x00,/*"C",35*/
0x00,0xFE,0x82,0x82,0x7C,0x00,/*"D",36*/
0x00,0xFE,0x92,0x92,0x82,0x00,/*"E",37*/
0x00,0xFE,0x90,0x90,0x80,0x00,/*"F",38*/
0x00,0x7C,0x82,0x92,0x5C,0x00,/*"G",39*/
0x00,0xFE,0x10,0x10,0xFE,0x00,/*"H",40*/
0x00,0x82,0xFE,0x82,0x00,0x00,/*"I",41*/
0x00,0x0C,0x02,0x02,0xFC,0x00,/*"J",42*/
0x00,0xFE,0x10,0x28,0xC6,0x00,/*"K",43*/
0x00,0xFE,0x02,0x02,0x02,0x00,/*"L",44*/
0x00,0xFE,0x60,0x60,0xFE,0x00,/*"M",45*/
0x00,0xFE,0x60,0x18,0xFE,0x00,/*"N",46*/
0x00,0xFE,0x82,0x82,0xFE,0x00,/*"O",47*/
0x00,0xFE,0x90,0x90,0x60,0x00,/*"P",48*/
0x00,0x7C,0x8A,0x86,0x7E,0x00,/*"Q",49*/
0x00,0xFE,0x98,0x94,0x62,0x00,/*"R",50*/
0x00,0x64,0x92,0x92,0x4C,0x00,/*"S",51*/
0x00,0x80,0xFE,0x80,0x80,0x00,/*"T",52*/
0x00,0xFC,0x02,0x02,0xFC,0x00,/*"U",53*/
0x00,0xF0,0x0E,0x0E,0xF0,0x00,/*"V",54*/
0x00,0xFE,0x0C,0x0C,0xFE,0x00,/*"W",55*/
0x00,0xC6,0x38,0x38,0xC6,0x00,/*"X",56*/
0xC0,0x20,0x1E,0x20,0xC0,0x00,/*"Y",57*/
0x00,0x86,0x9A,0xB2,0xC2,0x00,/*"Z",58*/
0x00,0x00,0xFE,0x82,0x00,0x00,/*"[",59*/
0xC0,0x30,0x0C,0x02,0x00,0x00,/*"\",60*/
0x00,0x00,0x82,0xFE,0x00,0x00,/*"]",61*/
0x00,0x40,0x80,0x40,0x00,0x00,/*"^",62*/
0x00,0x02,0x02,0x02,0x02,0x00,/*"_",63*/
0x00,0x00,0xC0,0x00,0x00,0x00,/*"`",64*/
0x00,0x04,0x1A,0x1A,0x1E,0x00,/*"a",65*/
0x00,0xFE,0x12,0x12,0x0C,0x00,/*"b",66*/
0x00,0x0C,0x12,0x12,0x12,0x00,/*"c",67*/
0x00,0x0C,0x12,0x12,0xFE,0x00,/*"d",68*/
0x00,0x0C,0x1A,0x1A,0x08,0x00,/*"e",69*/
0x00,0x10,0x3E,0x50,0x10,0x00,/*"f",70*/
0x00,0x08,0x15,0x15,0x1E,0x00,/*"g",71*/
0x00,0xFE,0x10,0x1E,0x00,0x00,/*"h",72*/
0x00,0x00,0x2E,0x00,0x00,0x00,/*"i",73*/
0x00,0x01,0x01,0x2E,0x00,0x00,/*"j",74*/
0x00,0xFE,0x08,0x14,0x12,0x00,/*"k",75*/
0x00,0x02,0xFE,0x02,0x00,0x00,/*"l",76*/
0x00,0x1E,0x10,0x1E,0x1E,0x00,/*"m",77*/
0x00,0x1E,0x10,0x10,0x0E,0x00,/*"n",78*/
0x00,0x0C,0x12,0x12,0x0C,0x00,/*"o",79*/
0x00,0x1F,0x12,0x12,0x0C,0x00,/*"p",80*/
0x00,0x0C,0x12,0x12,0x1F,0x00,/*"q",81*/
0x00,0x1E,0x08,0x10,0x10,0x00,/*"r",82*/
0x00,0x0A,0x1A,0x16,0x14,0x00,/*"s",83*/
0x00,0x10,0x3E,0x12,0x00,0x00,/*"t",84*/
0x00,0x1C,0x02,0x02,0x1E,0x00,/*"u",85*/
0x00,0x18,0x06,0x06,0x18,0x00,/*"v",86*/
0x18,0x06,0x1C,0x06,0x18,0x00,/*"w",87*/
0x00,0x12,0x0C,0x0C,0x12,0x00,/*"x",88*/
0x00,0x18,0x05,0x05,0x1E,0x00,/*"y",89*/
0x00,0x12,0x16,0x1A,0x12,0x00,/*"z",90*/
0x00,0x10,0xEE,0x82,0x00,0x00,/*"{",91*/
0x00,0x00,0xFE,0x00,0x00,0x00,/*"|",92*/
0x00,0x82,0xEE,0x10,0x00,0x00,/*"}",93*/
0x00,0x40,0x80,0x40,0x80,0x00,/*"~",94*/
};

const uint8_t EPD_Std_Font0612[] =
{
0x06,0x0C,0x20,0x5F, //12x12 Termianl PCtoLCD2002
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /*" ",0*/
0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0x60,0x00,0x00,0x00,0x00, /*"!",1*/
0x00,0x00,0x10,0x00,0x60,0x00,0x10,0x00,0x60,0x00,0x00,0x00, /*""",2*/
0x00,0x00,0x10,0x40,0x7F,0xF0,0x10,0x40,0x7F,0xF0,0x10,0x40, /*"#",3*/
0x00,0x00,0x18,0xC0,0x24,0x20,0x7F,0xF0,0x22,0x20,0x19,0xC0, /*"$",4*/
0x00,0x00,0x30,0x60,0x49,0x80,0x36,0xC0,0x19,0x20,0x60,0xC0, /*"%",5*/
0x00,0x00,0x33,0xC0,0x4C,0x20,0x33,0x20,0x00,0xC0,0x03,0x20, /*"&",6*/
0x00,0x00,0x00,0x00,0x50,0x00,0x60,0x00,0x00,0x00,0x00,0x00, /*"'",7*/
0x00,0x00,0x00,0x00,0x0F,0x80,0x30,0x60,0x40,0x10,0x00,0x00, /*"(",8*/
0x00,0x00,0x00,0x00,0x40,0x10,0x30,0x60,0x0F,0x80,0x00,0x00, /*")",9*/
0x00,0x00,0x0D,0x80,0x02,0x00,0x1F,0xC0,0x02,0x00,0x0D,0x80, /*"*",10*/
0x00,0x00,0x02,0x00,0x02,0x00,0x1F,0xC0,0x02,0x00,0x02,0x00, /*"+",11*/
0x00,0x00,0x00,0x00,0x00,0x50,0x00,0x60,0x00,0x00,0x00,0x00, /*",",12*/
0x00,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x00,0x00, /*"-",13*/
0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x60,0x00,0x00,0x00,0x00, /*".",14*/
0x00,0x00,0x00,0x30,0x00,0xC0,0x07,0x00,0x18,0x00,0x60,0x00, /*"/",15*/
0x00,0x00,0x1F,0xC0,0x20,0x20,0x20,0x20,0x1F,0xC0,0x00,0x00, /*"0",16*/
0x00,0x00,0x00,0x00,0x10,0x00,0x3F,0xE0,0x00,0x00,0x00,0x00, /*"1",17*/
0x00,0x00,0x18,0x60,0x21,0xA0,0x22,0x20,0x1C,0x20,0x00,0x00, /*"2",18*/
0x00,0x00,0x18,0xC0,0x22,0x20,0x22,0x20,0x1D,0xC0,0x00,0x00, /*"3",19*/
0x00,0x00,0x01,0x80,0x06,0x80,0x18,0x80,0x3F,0xE0,0x00,0x80, /*"4",20*/
0x00,0x00,0x3E,0xC0,0x24,0x20,0x24,0x20,0x23,0xC0,0x00,0x00, /*"5",21*/
0x00,0x00,0x1F,0xC0,0x22,0x20,0x22,0x20,0x19,0xC0,0x00,0x00, /*"6",22*/
0x00,0x00,0x20,0x00,0x20,0xE0,0x27,0x00,0x38,0x00,0x00,0x00, /*"7",23*/
0x00,0x00,0x1D,0xC0,0x22,0x20,0x22,0x20,0x1D,0xC0,0x00,0x00, /*"8",24*/
0x00,0x00,0x1C,0xC0,0x22,0x20,0x22,0x20,0x1F,0xC0,0x00,0x00, /*"9",25*/
0x00,0x00,0x00,0x00,0x0C,0x60,0x0C,0x60,0x00,0x00,0x00,0x00, /*":",26*/
0x00,0x00,0x00,0x00,0x0C,0x50,0x0C,0x60,0x00,0x00,0x00,0x00, /*";",27*/
0x00,0x00,0x02,0x00,0x05,0x00,0x08,0x80,0x10,0x40,0x20,0x20, /*"<",28*/
0x00,0x00,0x09,0x00,0x09,0x00,0x09,0x00,0x09,0x00,0x00,0x00, /*"=",29*/
0x00,0x00,0x20,0x20,0x10,0x40,0x08,0x80,0x05,0x00,0x02,0x00, /*">",30*/
0x00,0x00,0x18,0x00,0x20,0x00,0x23,0x60,0x1C,0x00,0x00,0x00, /*"?",31*/
0x00,0x00,0x1F,0xC0,0x29,0x20,0x2F,0x20,0x20,0xA0,0x1F,0x40, /*"@",32*/
0x00,0x00,0x01,0xE0,0x0E,0x80,0x30,0x80,0x0E,0x80,0x01,0xE0, /*"A",33*/
0x00,0x00,0x3F,0xE0,0x22,0x20,0x22,0x20,0x22,0x20,0x1D,0xC0, /*"B",34*/
0x00,0x00,0x1F,0xC0,0x20,0x20,0x20,0x20,0x20,0x20,0x18,0xC0, /*"C",35*/
0x00,0x00,0x3F,0xE0,0x20,0x20,0x20,0x20,0x10,0x40,0x0F,0x80, /*"D",36*/
0x00,0x00,0x3F,0xE0,0x22,0x20,0x22,0x20,0x22,0x20,0x20,0x20, /*"E",37*/
0x00,0x00,0x3F,0xE0,0x22,0x00,0x22,0x00,0x22,0x00,0x20,0x00, /*"F",38*/
0x00,0x00,0x1F,0xC0,0x20,0x20,0x20,0x20,0x21,0x40,0x19,0xE0, /*"G",39*/
0x00,0x00,0x3F,0xE0,0x02,0x00,0x02,0x00,0x02,0x00,0x3F,0xE0, /*"H",40*/
0x00,0x00,0x00,0x00,0x20,0x20,0x3F,0xE0,0x20,0x20,0x00,0x00, /*"I",41*/
0x00,0x00,0x00,0xC0,0x00,0x20,0x00,0x20,0x3F,0xC0,0x00,0x00, /*"J",42*/
0x00,0x00,0x3F,0xE0,0x02,0x00,0x0D,0x80,0x30,0x60,0x00,0x00, /*"K",43*/
0x00,0x00,0x3F,0xE0,0x00,0x20,0x00,0x20,0x00,0x20,0x00,0x20, /*"L",44*/
0x00,0x00,0x3F,0xE0,0x0E,0x00,0x01,0xE0,0x0E,0x00,0x3F,0xE0, /*"M",45*/
0x00,0x00,0x3F,0xE0,0x18,0x00,0x07,0x00,0x00,0xC0,0x3F,0xE0, /*"N",46*/
0x00,0x00,0x1F,0xC0,0x20,0x20,0x20,0x20,0x20,0x20,0x1F,0xC0, /*"O",47*/
0x00,0x00,0x3F,0xE0,0x22,0x00,0x22,0x00,0x22,0x00,0x1C,0x00, /*"P",48*/
0x00,0x00,0x1F,0xC0,0x20,0x20,0x20,0xA0,0x20,0x40,0x1F,0xA0, /*"Q",49*/
0x00,0x00,0x3F,0xE0,0x22,0x00,0x22,0x00,0x23,0x00,0x1C,0xE0, /*"R",50*/
0x00,0x00,0x18,0xC0,0x24,0x20,0x22,0x20,0x21,0x20,0x18,0xC0, /*"S",51*/
0x00,0x00,0x20,0x00,0x20,0x00,0x3F,0xE0,0x20,0x00,0x20,0x00, /*"T",52*/
0x00,0x00,0x3F,0xC0,0x00,0x20,0x00,0x20,0x00,0x20,0x3F,0xC0, /*"U",53*/
0x00,0x00,0x38,0x00,0x07,0x00,0x00,0xE0,0x07,0x00,0x38,0x00, /*"V",54*/
0x00,0x00,0x3E,0x00,0x01,0xE0,0x3E,0x00,0x01,0xE0,0x3E,0x00, /*"W",55*/
0x00,0x00,0x30,0x60,0x0D,0x80,0x02,0x00,0x0D,0x80,0x30,0x60, /*"X",56*/
0x00,0x00,0x30,0x00,0x0C,0x00,0x03,0xE0,0x0C,0x00,0x30,0x00, /*"Y",57*/
0x00,0x00,0x20,0x60,0x21,0xA0,0x22,0x20,0x2C,0x20,0x30,0x20, /*"Z",58*/
0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0xF0,0x40,0x10,0x40,0x10, /*"[",59*/
0x00,0x00,0x60,0x00,0x18,0x00,0x07,0x00,0x00,0xC0,0x00,0x30, /*"\",60*/
0x00,0x00,0x40,0x10,0x40,0x10,0x7F,0xF0,0x00,0x00,0x00,0x00, /*"]",61*/
0x00,0x00,0x00,0x00,0x20,0x00,0x40,0x00,0x20,0x00,0x00,0x00, /*"^",62*/
0x00,0x10,0x00,0x10,0x00,0x10,0x00,0x10,0x00,0x10,0x00,0x10, /*"_",63*/
0x00,0x00,0x00,0x00,0x40,0x00,0x20,0x00,0x00,0x00,0x00,0x00, /*"`",64*/
0x00,0x00,0x02,0xC0,0x05,0x20,0x05,0x20,0x03,0xC0,0x00,0x20, /*"a",65*/
0x00,0x00,0x3F,0xE0,0x04,0x20,0x04,0x20,0x04,0x20,0x03,0xC0, /*"b",66*/
0x00,0x00,0x03,0xC0,0x04,0x20,0x04,0x20,0x04,0x20,0x02,0x40, /*"c",67*/
0x00,0x00,0x03,0xC0,0x04,0x20,0x04,0x20,0x04,0x20,0x3F,0xE0, /*"d",68*/
0x00,0x00,0x03,0xC0,0x05,0x20,0x05,0x20,0x05,0x20,0x03,0x40, /*"e",69*/
0x00,0x00,0x04,0x00,0x1F,0xE0,0x24,0x00,0x20,0x00,0x00,0x00, /*"f",70*/
0x00,0x00,0x02,0xA0,0x05,0x50,0x05,0x50,0x02,0x50,0x04,0x20, /*"g",71*/
0x00,0x00,0x3F,0xE0,0x04,0x00,0x04,0x00,0x04,0x00,0x03,0xE0, /*"h",72*/
0x00,0x00,0x00,0x00,0x00,0x00,0x37,0xE0,0x00,0x00,0x00,0x00, /*"i",73*/
0x00,0x00,0x00,0x10,0x00,0x10,0x37,0xE0,0x00,0x00,0x00,0x00, /*"j",74*/
0x00,0x00,0x3F,0xE0,0x00,0x80,0x01,0x80,0x02,0x40,0x04,0x20, /*"k",75*/
0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0xE0,0x00,0x00,0x00,0x00, /*"l",76*/
0x00,0x00,0x07,0xE0,0x04,0x00,0x03,0xE0,0x04,0x00,0x03,0xE0, /*"m",77*/
0x00,0x00,0x07,0xE0,0x04,0x00,0x04,0x00,0x04,0x00,0x03,0xE0, /*"n",78*/
0x00,0x00,0x03,0xC0,0x04,0x20,0x04,0x20,0x04,0x20,0x03,0xC0, /*"o",79*/
0x00,0x00,0x07,0xF0,0x04,0x40,0x04,0x40,0x04,0x40,0x03,0x80, /*"p",80*/
0x00,0x00,0x03,0x80,0x04,0x40,0x04,0x40,0x04,0x40,0x07,0xF0, /*"q",81*/
0x00,0x00,0x00,0x00,0x07,0xE0,0x02,0x00,0x04,0x00,0x04,0x00, /*"r",82*/
0x00,0x00,0x02,0x40,0x05,0x20,0x05,0x20,0x04,0xA0,0x02,0x40, /*"s",83*/
0x00,0x00,0x04,0x00,0x3F,0xC0,0x04,0x20,0x00,0x20,0x00,0x00, /*"t",84*/
0x00,0x00,0x07,0xC0,0x00,0x20,0x00,0x20,0x00,0x20,0x07,0xE0, /*"u",85*/
0x00,0x00,0x06,0x00,0x01,0x80,0x00,0x60,0x01,0x80,0x06,0x00, /*"v",86*/
0x00,0x00,0x07,0x00,0x00,0xE0,0x07,0x00,0x00,0xE0,0x07,0x00, /*"w",87*/
0x00,0x00,0x04,0x20,0x02,0x40,0x01,0x80,0x02,0x40,0x04,0x20, /*"x",88*/
0x00,0x00,0x06,0x10,0x01,0x90,0x00,0x60,0x01,0x80,0x06,0x00, /*"y",89*/
0x00,0x00,0x04,0x20,0x04,0x60,0x04,0xA0,0x05,0x20,0x06,0x20, /*"z",90*/
0x00,0x00,0x00,0x00,0x02,0x00,0x7D,0xF0,0x40,0x10,0x00,0x00, /*"{",91*/
0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xF0,0x00,0x00,0x00,0x00, /*"|",92*/
0x00,0x00,0x00,0x00,0x40,0x10,0x7D,0xF0,0x02,0x00,0x00,0x00, /*"}",93*/
0x00,0x00,0x20,0x00,0x40,0x00,0x40,0x00,0x20,0x00,0x40,0x00, /*"~",94*/
};

const uint8_t EPD_L_Font0816[] = 
{
0x08,0x10,0x20,0x5F, //16x16 FixedSystem PCtoLCD2002
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,/*" ",0*/
0x00,0x00,0x00,0x00,0x0E,0x00,0x1F,0xB0,0x1F,0xB0,0x0E,0x00,0x00,0x00,0x00,0x00,/*"!",1*/
0x00,0x00,0x1C,0x00,0x1C,0x00,0x00,0x00,0x00,0x00,0x1C,0x00,0x1C,0x00,0x00,0x00,/*""",2*/
0x00,0x00,0x04,0x40,0x1F,0xF0,0x1F,0xF0,0x04,0x40,0x1F,0xF0,0x1F,0xF0,0x04,0x40,/*"#",3*/
0x00,0x00,0x0C,0x20,0x1E,0x30,0x73,0x1C,0x71,0x9C,0x18,0xF0,0x08,0x60,0x00,0x00,/*"$",4*/
0x18,0x00,0x3C,0x60,0x24,0xC0,0x3D,0xB0,0x1B,0x78,0x06,0x48,0x0C,0x78,0x00,0x30,/*"%",5*/
0x00,0x00,0x0D,0xE0,0x1F,0xF0,0x12,0x10,0x1E,0x90,0x0C,0xE0,0x00,0xF0,0x00,0x90,/*"&",6*/
0x00,0x00,0x00,0x00,0x00,0x00,0x1C,0x00,0x1C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,/*"'",7*/
0x00,0x00,0x00,0x00,0x03,0xE0,0x0F,0xF8,0x1C,0x1C,0x10,0x04,0x00,0x00,0x00,0x00,/*"(",8*/
0x00,0x00,0x00,0x00,0x10,0x04,0x1C,0x1C,0x0F,0xF8,0x03,0xE0,0x00,0x00,0x00,0x00,/*")",9*/
0x00,0x00,0x01,0x00,0x05,0x40,0x07,0xC0,0x03,0x80,0x07,0xC0,0x05,0x40,0x01,0x00,/*"*",10*/
0x00,0x00,0x01,0x00,0x01,0x00,0x07,0xC0,0x07,0xC0,0x01,0x00,0x01,0x00,0x00,0x00,/*"+",11*/
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x00,0x3C,0x00,0x38,0x00,0x00,0x00,0x00,/*",",12*/
0x00,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x00,0x00,/*"-",13*/
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x00,0x00,0x00,/*".",14*/
0x00,0x00,0x00,0x18,0x00,0x78,0x01,0xE0,0x07,0x80,0x1E,0x00,0x18,0x00,0x00,0x00,/*"/",15*/
0x00,0x00,0x00,0x00,0x0F,0xE0,0x1F,0xF0,0x10,0xD0,0x16,0x10,0x1F,0xF0,0x0F,0xE0,/*"0",16*/
0x00,0x00,0x04,0x00,0x04,0x00,0x0C,0x00,0x1F,0xF0,0x1F,0xF0,0x00,0x00,0x00,0x00,/*"1",17*/
0x00,0x00,0x0C,0x30,0x1C,0x70,0x10,0xD0,0x11,0x90,0x1F,0x10,0x0E,0x10,0x00,0x00,/*"2",18*/
0x00,0x00,0x0C,0x60,0x1C,0x70,0x11,0x10,0x11,0x10,0x1F,0xF0,0x0E,0xE0,0x00,0x00,/*"3",19*/
0x00,0x00,0x00,0xC0,0x1F,0xC0,0x1F,0x40,0x00,0x40,0x07,0xF0,0x07,0xF0,0x00,0x40,/*"4",20*/
0x00,0x00,0x1F,0x10,0x1F,0x10,0x11,0x10,0x11,0x30,0x11,0xE0,0x10,0xC0,0x00,0x00,/*"5",21*/
0x00,0x00,0x03,0xE0,0x07,0xF0,0x1E,0x10,0x1A,0x10,0x13,0xF0,0x01,0xE0,0x00,0x00,/*"6",22*/
0x00,0x00,0x10,0x00,0x10,0x70,0x11,0xF0,0x17,0x80,0x1E,0x00,0x18,0x00,0x00,0x00,/*"7",23*/
0x00,0x00,0x0E,0xE0,0x1F,0xF0,0x13,0x10,0x11,0x90,0x1F,0xF0,0x0E,0xE0,0x00,0x00,/*"8",24*/
0x00,0x00,0x0F,0x00,0x1F,0x90,0x10,0xB0,0x10,0xF0,0x1F,0xC0,0x0F,0x80,0x00,0x00,/*"9",25*/
0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x30,0x06,0x30,0x06,0x30,0x00,0x00,0x00,0x00,/*":",26*/
0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x34,0x06,0x3C,0x06,0x38,0x00,0x00,0x00,0x00,/*";",27*/
0x00,0x00,0x01,0x00,0x03,0x80,0x06,0xC0,0x0C,0x60,0x18,0x30,0x10,0x10,0x00,0x00,/*"<",28*/
0x00,0x00,0x02,0x80,0x02,0x80,0x02,0x80,0x02,0x80,0x02,0x80,0x02,0x80,0x00,0x00,/*"=",29*/
0x00,0x00,0x10,0x10,0x18,0x30,0x0C,0x60,0x06,0xC0,0x03,0x80,0x01,0x00,0x00,0x00,/*">",30*/
0x00,0x00,0x0C,0x00,0x1C,0x00,0x11,0xB0,0x13,0xB0,0x1E,0x00,0x0C,0x00,0x00,0x00,/*"?",31*/
0x0F,0xE0,0x1F,0xF0,0x10,0x10,0x11,0x90,0x13,0xD0,0x12,0x50,0x1F,0xD0,0x0F,0xD0,/*"@",32*/
0x00,0x00,0x07,0xF0,0x0F,0xF0,0x18,0x80,0x18,0x80,0x0F,0xF0,0x07,0xF0,0x00,0x00,/*"A",33*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x11,0x10,0x11,0x10,0x1F,0xF0,0x0E,0xE0,0x00,0x00,/*"B",34*/
0x00,0x00,0x0F,0xE0,0x1F,0xF0,0x10,0x10,0x10,0x10,0x1C,0x70,0x0C,0x60,0x00,0x00,/*"C",35*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x10,0x10,0x18,0x30,0x0F,0xE0,0x07,0xC0,0x00,0x00,/*"D",36*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x11,0x10,0x11,0x10,0x11,0x10,0x10,0x10,0x00,0x00,/*"E",37*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x11,0x00,0x11,0x00,0x11,0x00,0x10,0x00,0x00,0x00,/*"F",38*/
0x00,0x00,0x0F,0xE0,0x1F,0xF0,0x10,0x10,0x10,0x90,0x1C,0xF0,0x0C,0xF0,0x00,0x00,/*"G",39*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x01,0x00,0x01,0x00,0x1F,0xF0,0x1F,0xF0,0x00,0x00,/*"H",40*/
0x00,0x00,0x00,0x00,0x10,0x10,0x1F,0xF0,0x1F,0xF0,0x10,0x10,0x00,0x00,0x00,0x00,/*"I",41*/
0x00,0x00,0x00,0x60,0x00,0x70,0x00,0x10,0x00,0x10,0x1F,0xF0,0x1F,0xE0,0x00,0x00,/*"J",42*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x01,0x00,0x07,0xC0,0x1E,0xF0,0x18,0x30,0x00,0x00,/*"K",43*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x00,0x10,0x00,0x10,0x00,0x10,0x00,0x10,0x00,0x00,/*"L",44*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x04,0x00,0x03,0x80,0x04,0x00,0x1F,0xF0,0x1F,0xF0,/*"M",45*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x06,0x00,0x03,0x00,0x01,0x80,0x1F,0xF0,0x1F,0xF0,/*"N",46*/
0x00,0x00,0x0F,0xE0,0x1F,0xF0,0x10,0x10,0x10,0x10,0x1F,0xF0,0x0F,0xE0,0x00,0x00,/*"O",47*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x11,0x00,0x11,0x00,0x1F,0x00,0x0E,0x00,0x00,0x00,/*"P",48*/
0x00,0x00,0x0F,0xE0,0x1F,0xF0,0x10,0x10,0x10,0x18,0x1F,0xFC,0x0F,0xE4,0x00,0x00,/*"Q",49*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x11,0x00,0x11,0x80,0x1F,0xF0,0x0E,0x70,0x00,0x00,/*"R",50*/
0x00,0x00,0x0C,0x20,0x1E,0x30,0x13,0x10,0x11,0x90,0x18,0xF0,0x08,0x60,0x00,0x00,/*"S",51*/
0x00,0x00,0x10,0x00,0x10,0x00,0x1F,0xF0,0x1F,0xF0,0x10,0x00,0x10,0x00,0x00,0x00,/*"T",52*/
0x00,0x00,0x1F,0xE0,0x1F,0xF0,0x00,0x10,0x00,0x10,0x1F,0xF0,0x1F,0xE0,0x00,0x00,/*"U",53*/
0x00,0x00,0x1F,0xC0,0x1F,0xE0,0x00,0x30,0x00,0x30,0x1F,0xE0,0x1F,0xC0,0x00,0x00,/*"V",54*/
0x00,0x00,0x1F,0x80,0x1F,0xF0,0x00,0x70,0x03,0x80,0x00,0x70,0x1F,0xF0,0x1F,0x80,/*"W",55*/
0x00,0x00,0x18,0x70,0x1C,0xF0,0x07,0x00,0x03,0x80,0x1C,0xF0,0x18,0x70,0x00,0x00,/*"X",56*/
0x00,0x00,0x1E,0x00,0x1F,0x00,0x01,0xF0,0x01,0xF0,0x1F,0x00,0x1E,0x00,0x00,0x00,/*"Y",57*/
0x00,0x00,0x10,0x70,0x10,0xF0,0x11,0x90,0x13,0x10,0x1E,0x10,0x1C,0x10,0x00,0x00,/*"Z",58*/
0x00,0x00,0x00,0x00,0x1F,0xFE,0x1F,0xFE,0x10,0x02,0x10,0x02,0x00,0x00,0x00,0x00,/*"[",59*/
0x00,0x00,0x18,0x00,0x1E,0x00,0x07,0x80,0x01,0xE0,0x00,0x78,0x00,0x18,0x00,0x00,/*"\",60*/
0x00,0x00,0x00,0x00,0x10,0x02,0x10,0x02,0x1F,0xFE,0x1F,0xFE,0x00,0x00,0x00,0x00,/*"]",61*/
0x00,0x00,0x10,0x00,0x30,0x00,0x60,0x00,0x60,0x00,0x30,0x00,0x10,0x00,0x00,0x00,/*"^",62*/
0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,/*"_",63*/
0x00,0x00,0x00,0x00,0x40,0x00,0x60,0x00,0x70,0x00,0x10,0x00,0x00,0x00,0x00,0x00,/*"`",64*/
0x00,0x00,0x00,0x60,0x04,0xF0,0x04,0x90,0x04,0x90,0x07,0xF0,0x03,0xF0,0x00,0x00,/*"a",65*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x04,0x10,0x04,0x10,0x07,0xF0,0x03,0xE0,0x00,0x00,/*"b",66*/
0x00,0x00,0x03,0xE0,0x07,0xF0,0x04,0x10,0x04,0x10,0x06,0x30,0x02,0x20,0x00,0x00,/*"c",67*/
0x00,0x00,0x03,0xE0,0x07,0xF0,0x04,0x10,0x04,0x10,0x1F,0xF0,0x1F,0xF0,0x00,0x00,/*"d",68*/
0x00,0x00,0x03,0xE0,0x07,0xF0,0x04,0x90,0x04,0x90,0x07,0x90,0x03,0x80,0x00,0x00,/*"e",69*/
0x00,0x00,0x01,0x00,0x0F,0xF0,0x1F,0xF0,0x11,0x00,0x11,0x00,0x11,0x00,0x00,0x00,/*"f",70*/
0x00,0x00,0x03,0xE2,0x07,0xF2,0x04,0x12,0x04,0x12,0x07,0xFE,0x07,0xFC,0x00,0x00,/*"g",71*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x04,0x00,0x04,0x00,0x07,0xF0,0x03,0xF0,0x00,0x00,/*"h",72*/
0x00,0x00,0x04,0x10,0x04,0x10,0x37,0xF0,0x37,0xF0,0x00,0x10,0x00,0x10,0x00,0x00,/*"i",73*/
0x00,0x00,0x00,0x02,0x04,0x02,0x04,0x02,0x37,0xFE,0x37,0xFC,0x00,0x00,0x00,0x00,/*"j",74*/
0x00,0x00,0x1F,0xF0,0x1F,0xF0,0x00,0x80,0x01,0xC0,0x07,0x70,0x06,0x30,0x00,0x00,/*"k",75*/
0x00,0x00,0x10,0x10,0x10,0x10,0x1F,0xF0,0x1F,0xF0,0x00,0x10,0x00,0x10,0x00,0x00,/*"l",76*/
0x00,0x00,0x07,0xF0,0x07,0xF0,0x04,0x00,0x07,0xE0,0x04,0x00,0x07,0xF0,0x03,0xF0,/*"m",77*/
0x00,0x00,0x07,0xF0,0x07,0xF0,0x04,0x00,0x04,0x00,0x07,0xF0,0x03,0xF0,0x00,0x00,/*"n",78*/
0x00,0x00,0x03,0xE0,0x07,0xF0,0x04,0x10,0x04,0x10,0x07,0xF0,0x03,0xE0,0x00,0x00,/*"o",79*/
0x00,0x00,0x07,0xFE,0x07,0xFE,0x04,0x10,0x04,0x10,0x07,0xF0,0x03,0xE0,0x00,0x00,/*"p",80*/
0x00,0x00,0x03,0xE0,0x07,0xF0,0x04,0x10,0x04,0x10,0x07,0xFE,0x07,0xFE,0x00,0x00,/*"q",81*/
0x00,0x00,0x07,0xF0,0x07,0xF0,0x01,0x00,0x02,0x00,0x06,0x00,0x06,0x00,0x00,0x00,/*"r",82*/
0x00,0x00,0x03,0x10,0x07,0x90,0x04,0x90,0x04,0x90,0x04,0xF0,0x04,0x60,0x00,0x00,/*"s",83*/
0x00,0x00,0x04,0x00,0x1F,0xE0,0x1F,0xF0,0x04,0x10,0x04,0x10,0x04,0x10,0x00,0x00,/*"t",84*/
0x00,0x00,0x07,0xE0,0x07,0xF0,0x00,0x10,0x00,0x10,0x07,0xF0,0x07,0xF0,0x00,0x00,/*"u",85*/
0x00,0x00,0x07,0xC0,0x07,0xE0,0x00,0x30,0x00,0x30,0x07,0xE0,0x07,0xC0,0x00,0x00,/*"v",86*/
0x00,0x00,0x07,0xC0,0x07,0xF0,0x00,0x30,0x03,0xC0,0x00,0x30,0x07,0xF0,0x07,0xC0,/*"w",87*/
0x00,0x00,0x06,0x30,0x07,0x70,0x01,0xC0,0x01,0xC0,0x07,0x70,0x06,0x30,0x00,0x00,/*"x",88*/
0x00,0x02,0x07,0xE2,0x07,0xF2,0x00,0x16,0x00,0x1C,0x07,0xF8,0x07,0xE0,0x00,0x00,/*"y",89*/
0x00,0x00,0x04,0x30,0x04,0x70,0x04,0xD0,0x05,0x90,0x07,0x10,0x06,0x10,0x00,0x00,/*"z",90*/
0x00,0x00,0x00,0x80,0x01,0xC0,0x0F,0x78,0x1E,0x3C,0x10,0x04,0x00,0x00,0x00,0x00,/*"{",91*/
0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0xFE,0x1F,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,/*"|",92*/
0x00,0x00,0x00,0x00,0x10,0x04,0x1E,0x3C,0x0F,0x78,0x01,0xC0,0x00,0x80,0x00,0x00,/*"}",93*/
0x0C,0x00,0x18,0x00,0x10,0x00,0x18,0x00,0x0C,0x00,0x04,0x00,0x0C,0x00,0x18,0x00,/*"~",94*/
};

/* Used for price */
const uint8_t EPD_XL_Font1224[] = 
{
0x0C,0x18,'.','9', //24x24, HeiTi, from ".", "0"~"9"
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x00,0x18,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,/*".",14*/
0x00,0x00,0x04,0x00,0x00,0x18,0x00,0x00,0x60,0x00,0x01,0x80,0x00,0x06,0x00,0x00,
0x08,0x00,0x00,0x30,0x00,0x00,0xC0,0x00,0x03,0x00,0x00,0x0C,0x00,0x00,0x10,0x00,
0x00,0x00,0x00,0x00,/*"/",15*/
0x00,0x00,0x00,0x00,0x7F,0x80,0x01,0xFF,0xE0,0x03,0x80,0x70,0x07,0x00,0x38,0x06,
0x00,0x18,0x06,0x00,0x18,0x06,0x00,0x38,0x03,0x80,0x70,0x01,0xFF,0xE0,0x00,0x7F,
0x80,0x00,0x00,0x00,/*"0",16*/
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x80,0x00,0x01,0x80,0x00,0x03,
0xFF,0xF8,0x07,0xFF,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,/*"1",17*/
0x00,0x00,0x00,0x00,0xE0,0x38,0x03,0xE0,0x78,0x07,0x00,0xD8,0x06,0x01,0x98,0x06,
0x03,0x18,0x06,0x06,0x18,0x07,0x0C,0x18,0x03,0xF8,0x18,0x01,0xF0,0x18,0x00,0x00,
0x00,0x00,0x00,0x00,/*"2",18*/
0x00,0x00,0x00,0x00,0xC0,0xE0,0x03,0xC0,0xF0,0x03,0x00,0x30,0x06,0x00,0x18,0x06,
0x18,0x18,0x06,0x18,0x18,0x06,0x18,0x18,0x07,0x3C,0x30,0x03,0xE7,0xF0,0x01,0xC3,
0xC0,0x00,0x00,0x00,/*"3",19*/
0x00,0x00,0x00,0x00,0x03,0x80,0x00,0x0F,0x80,0x00,0x1F,0x80,0x00,0x79,0x80,0x01,
0xE1,0x80,0x03,0xC1,0x80,0x07,0xFF,0xF8,0x07,0xFF,0xF8,0x00,0x01,0x80,0x00,0x01,
0x80,0x00,0x00,0x00,/*"4",20*/
0x00,0x00,0x00,0x00,0x00,0xC0,0x00,0x78,0xF0,0x07,0xF8,0x30,0x07,0x98,0x18,0x06,
0x30,0x18,0x06,0x30,0x18,0x06,0x30,0x18,0x06,0x38,0x30,0x06,0x1F,0xF0,0x06,0x0F,
0xC0,0x00,0x00,0x00,/*"5",21*/
0x00,0x00,0x00,0x00,0x7F,0x80,0x01,0xFF,0xE0,0x03,0x98,0x30,0x06,0x30,0x18,0x06,
0x30,0x18,0x06,0x30,0x18,0x07,0x38,0x38,0x03,0x9F,0xF0,0x01,0x8F,0xC0,0x00,0x00,
0x00,0x00,0x00,0x00,/*"6",22*/
0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x06,0x00,0x00,0x06,0x00,0x00,0x06,
0x00,0x18,0x06,0x00,0xF8,0x06,0x0F,0xE0,0x06,0x7F,0x00,0x07,0xF0,0x00,0x07,0x80,
0x00,0x00,0x00,0x00,/*"7",23*/
0x00,0x00,0x00,0x01,0xE1,0xE0,0x03,0xF3,0xF0,0x03,0x1E,0x38,0x06,0x0C,0x18,0x06,
0x0C,0x18,0x06,0x0C,0x18,0x06,0x0C,0x18,0x03,0x1E,0x38,0x03,0xF3,0xF0,0x01,0xE1,
0xE0,0x00,0x00,0x00,/*"8",24*/
0x00,0x00,0x00,0x00,0xFC,0x60,0x03,0xFE,0x70,0x07,0x07,0x38,0x06,0x03,0x18,0x06,
0x03,0x18,0x06,0x03,0x18,0x03,0x06,0x70,0x03,0xFF,0xE0,0x00,0xFF,0x80,0x00,0x00,
0x00,0x00,0x00,0x00,/*"9",25*/
};

const uint8_t EPD_2bpp_LUT[] =
{   
  0x00,// 0000 -> 0000-0000   
  0x03,// 0001 -> 0000-0011   
  0x0C,// 0010 -> 0000-1100   
  0x0F,// 0011 -> 0000-1111   
  0x30,// 0100 -> 0011-0000   
  0x33,// 0101 -> 0011-0011   
  0x3C,// 0110 -> 0011-1100   
  0x3F,// 0111 -> 0011-1111   
  0xC0,// 1000 -> 1100-0000  
  0xC3,// 1001 -> 1100-0011  
  0xCC,// 1010 -> 1100-1100  
  0xCF,// 1011 -> 1100-1111  
  0xF0,// 1100 -> 1111-0000   
  0xF3,// 1101 -> 1111-0011  
  0xFC,// 1110 -> 1111-1100  
  0xFF,// 1111 -> 1111-1111 
};

#endif
