/*
    Author: Allan K Liu
    Original: Athanassios Mavrogeorgiadis
    
    Changes:
    - Optimization for Low Power Consumption Operations
    - GD0 as external interrupt, to bring MCU from deep sleep mode.
    - GPIO setup
    - Clock setup
    - Event driven design
    - ALOHA transceiver for channel access with RTS/CTS/ACK
    - S-MAC
    - Cloned high level designs from SimpliciTI.
    - New CircularBuffer from mbed 5
    
    Status:
    - CC1101 SPI access to regsiters from PARTNUM/VERSION to RCCTRL0 not stable
    - PARTNUM ranges from 0x00/0xFF
    - VERSION ranges from 0x0F/0x1F/0x4F
 */
#include "mbed.h"
#include "CC1101.h"
#include "RingBuffer.h"
 

#if defined(TARGET_NUCLEO_F103RB) || defined(TARGET_LPC824)
// RDmiso is actually the MISO, 
// SPI only works after MISO turns into low from Hi-Z state of CC1101
// Therefore we need a dedicated input to detect.
CC1101 cc1101(D11, D12, D13, D10, D8);    // MOSI, MISO, SCK, nCS, RDmiso
DigitalIn gdo0(D7);     // InterruptIn is better than DigitalIn
InterruptIn gdo2(D9);   // InterruptIn is better than DigitalIn

DigitalOut led1(A0);  // timer blink led
DigitalOut led2(A1);  // RX led
DigitalOut led3(A2);  // TX led

#elif defined(TARGET_LPC1114)
// Legacy platform, to be updated
CC1101 cc1101(p5, p6, p7, p8, p10);
DigitalIn gdo0(p9);     // pin connected to gdo0 pin of CC1101 for checking that received a new packet

DigitalOut led1(LED1);  // timer blink led
DigitalOut led2(LED2);  // RX led
DigitalOut led3(LED3);  // TX led

#else

#warning "YOU HAVE TO DEFINE A H/W."
#endif


Ticker timer;


Serial pc(USBTX, USBRX); // tx, rx, to be replaced by USB CDC interface in a hub/sink node.
RingBuffer rbRX(512);   // ring buffer for the pc RX data
RingBuffer rbTX(512);   // ring buffer for the pc TX data

Timeout rbRXtimeout;
Timeout led2timeout;
Timeout led3timeout;

unsigned char buffer[128];

static unsigned char sil = 0;
static unsigned char ver = 0;
static unsigned char sta = 0;
static unsigned char val = 0;

void led2timeout_func()
{
    led2 = 0;
    led2timeout.detach();
}

void led3timeout_func()
{
    led3 = 0;
    led3timeout.detach();
}

void rbRXtimeout_func()         // function for transmiting the RF packets - empty the rbRX ring buffer
{
    unsigned char txlength;
    
    txlength = 0;
    while(rbRX.use() > 0)
    {
      led2 = 1;
      buffer[txlength] = rbRX.getc();
      txlength++;
      led2timeout.attach(&led2timeout_func, 0.050);  // for switch off the led
    }
    if (txlength)
      cc1101.SendPacket(buffer, txlength);    // tx packet
    
    rbRXtimeout.detach();
}

void timer_func()           // check the status of the CC1101 every 100ms
{    
    unsigned char chip_status_rx, chip_status_tx;
    
    led1 = !led1;
    chip_status_rx = cc1101.ReadChipStatusRX();  // check the rx status 
    if ((chip_status_rx & CHIP_STATE_MASK) == CHIP_STATE_RXFIFO_OVERFLOW)   // if rx overflow flush the rx fifo
      cc1101.FlushRX();  
    if ((chip_status_rx & CHIP_STATE_MASK) == CHIP_STATE_IDLE)              // if state is idle go to rx state again
      cc1101.RXMode();  
    chip_status_tx = cc1101.ReadChipStatusTX();  // check the tx sttus
    if ((chip_status_tx & CHIP_STATE_MASK) == CHIP_STATE_TXFIFO_UNDERFLOW)  // if tx underflow flush the tx fifo
      cc1101.FlushTX();
}

int main() 
{
    unsigned char rxlength, i;
        unsigned char buf[128];
    
    rbRX.clear();
    rbTX.clear();
    cc1101.init();

        // test routines

#define REG_PART_DBG 1  
#if defined(REG_PART_DBG)
    
        for (int i=0; i<10; i++){
            buf[i] = cc1101.ReadChipStatusRX();
        }
        cc1101.ReadBurstReg(CCxxx0_PARTNUM, buf, CCxxx0_PATABLE-CCxxx0_PARTNUM+1);

    // Read them one by one
        for (int i=CCxxx0_PARTNUM; i<CCxxx0_PATABLE; i++){
            buf[i] = cc1101.ReadReg(i);
        }
    
        // Try to read back all of values from registers ranges from 0x30~0x3D
        // Read them one by one
        sil = cc1101.ReadReg(CCxxx0_PARTNUM);
      ver = cc1101.ReadReg(CCxxx0_VERSION);
        sta = cc1101.ReadChipStatusRX();
    
        for (int i=0; i<10; i++){
            buf[i] = cc1101.ReadReg(CCxxx0_PARTNUM);
        }

        for (int i=0; i<10; i++){
            buf[i] = cc1101.ReadReg(CCxxx0_VERSION);
        }

        for (int i=0; i<10; i++){
            buf[i] = cc1101.ReadChipStatusRX();
        }

        // Read them in a burst reading
        cc1101.ReadBurstReg(CCxxx0_PARTNUM, buf, CCxxx0_PATABLE-CCxxx0_PARTNUM+1);
        // Read them in a burst reading twice
        cc1101.ReadBurstReg(CCxxx0_PARTNUM, buf, CCxxx0_PATABLE-CCxxx0_PARTNUM+1);
        // Read them one by one
        for (int i=CCxxx0_PARTNUM; i<CCxxx0_PATABLE; i++){
            buf[i] = cc1101.ReadReg(i);
        }
        // Read them in a burst reading
        cc1101.ReadBurstReg(CCxxx0_PARTNUM, buf, CCxxx0_PATABLE-CCxxx0_PARTNUM+1);
        
#endif

//#define REG_RW_DBG 1
#if defined(REG_RW_DBG)
        // Try to read back all of values from registers ranges from 0x00~0x2E
        cc1101.ReadBurstReg(CCxxx0_IOCFG2, buf, CCxxx0_TEST0+1);
        
        for (int i=CCxxx0_IOCFG2; i<(CCxxx0_TEST0+1); i++){
            buf[i] = cc1101.ReadReg(i);
        }
        
        const unsigned char params[0x2F] = \
            {0x06,0x2E,0x07,0x07,0xD3,0x91,0xFF,0x04, \
             0x05,0x00,0x00,0x06,0x00,0x10,0xB1,0x3B, \
             0xF8,0x83,0x13,0x22,0xF8,0x15,0x07,0x3F, \
             0x18,0x16,0x6C,0x03,0x40,0x91,0x87,0x6B, \
             0xF8,0x56,0x10,0xE9,0x2A,0x00,0x1F,0x41, \
             0x00,0x59,0x7F,0x63,0x88,0x31,0x09};
            
        for (int i=CCxxx0_IOCFG2; i<(CCxxx0_TEST0+1); i++){
            cc1101.WriteReg(i, params[i]);
        }

        for (int i=CCxxx0_IOCFG2; i<(CCxxx0_TEST0+1); i++){
            buf[i] = cc1101.ReadReg(i);
        }

    cc1101.ReadBurstReg(CCxxx0_IOCFG2, buf, CCxxx0_TEST0+1);

#endif      
        
        // end of test
    
    timer.attach(&timer_func, 0.1);
    while(1)
    {
        if(gdo0)      // rx finished and CRC OK read the new packet
        {
            rxlength = sizeof(buffer);
            if (cc1101.ReceivePacket(buffer, &rxlength) == 1)   // read the rx packet
            {
                led3 = 1;
                for (i = 0; i < rxlength; i++)
                    rbTX.putc(buffer[i]);                       // store the packet to the rbTX ring buffer 
                led3timeout.attach(&led3timeout_func, 0.050);   // for switch off the led
            }
        }
        if (rbTX.use() > 0)         // check if we have data to transmit to pc
            pc.putc(rbTX.getc());   // get the data from the ring buffer and transmit it to the pc
        if (pc.readable())          // check if we received new data from the pc
        {
            rbRX.putc(pc.getc());   // put the data to the rbRX buffer and wait until 20ms passed till the last byte before tx the packet in RF
            rbRXtimeout.attach(&rbRXtimeout_func, 0.020);
        }
        if (rbRX.use() > 20)        // if more than 20 bytes received then tx the packet in RF
            rbRXtimeout_func();
    }     
}
