#include "mbed.h"
#include "BLEDevice.h"
#include "BroadcasterService.h"
#include "ErrorService.h"
#include <string>

using namespace std;

BLEDevice ble;
DigitalOut led1(LED1);
InterruptIn indicatorIn(p5);
Serial pc(USBTX, USBRX);

/*Variable Declarations*/
const static char       DEVICE_NAME[]       = "BLE_Broadcaster";
static volatile bool    indicatorOn         = false;
static volatile bool    indicatorSent       = false;
uint8_t                 cmdIndicatorOn[8]   = { 0x69,0x6e,0x64,0x69,0xff,0xff,0xff,0xff }; // = I   N   D   I   255 255 255 255
uint8_t                 cmdIndicatorOff[8]  = { 0x69,0x6e,0x64,0x69,0x00,0x00,0x00,0x00 }; // = I   N   D   I   0   0   0   0 
static const uint16_t uuid16_list[]        = {BroadcasterService::BROADCAST_SERVICE_UUID};

/** Callback function for ticker */
void blink(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */  
}

/** Callback routine is interrupt activated by a debounced button hit*/
void indicatorSwitchOn(void) {
    indicatorOn = true;   
    indicatorSent = false;
    pc.printf("Indicator switch on.\r\n");
}

/** Callback routine is interrupt activated by a debounced button release*/
void indicatorSwitchOff(void) {
    indicatorOn = false;   
    indicatorSent = false;
    pc.printf("Indicator switch off.\r\n");
}

/** Callback function for BLE disconnection */
void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    pc.printf("Disconnected! - start advertising. Handle:%d, Reason:0x%02x\r\n", handle, reason);
    ble.startAdvertising();
}

/** Callback function for BLE connection */
void connectionCallback(Gap::Handle_t handle, Gap::addr_type_t peerAddrType, const Gap::address_t peerAddr, const Gap::ConnectionParams_t *parms)
{
    pc.printf("Connected! - stop advertising. Handle:%d, eType:%d, Addr:%u.\r\n", handle, peerAddrType, peerAddr);
    ble.stopAdvertising();
}

int main(void)
{    
    /*Setup switch input*/
    indicatorIn.mode(PullUp);
    // Delay for initial pullup to take effect
    wait(.01);    
    // Setup Interrupt callback function for a button hit
    indicatorIn.fall(&indicatorSwitchOn);
    indicatorIn.rise(&indicatorSwitchOff);
    
    //TODO: Initial light input - if switch is ON at program start
    
    /*Setup Blinky*/
    led1 = 1;
    Ticker t;
    t.attach(&blink, 1);

    /*Setup BLE*/
    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onConnection(connectionCallback);

    //create services  
    BroadcasterService broadcasterService(ble);
    ErrorService errorService(ble);

    /*
    * BREDR_NOT_SUPPORTED = BLE only
    * LE_GENERAL_DISCOVERABLE = Device is discoverable at any moment (no time out)
    * ADV_CONNECTABLE_UNDIRECTED = Any central device can connect
    */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    
    ble.setAdvertisingInterval(1600); /* 1s; in multiples of 0.625ms. */
    ble.startAdvertising();

    pc.printf("Advertising node %s\n\r", DEVICE_NAME);

    //loop forever
    while(true) {
        ble.waitForEvent(); // this should return upon any system event (such as an interrupt or a ticker wakeup)        
        
        if(indicatorOn) { //if button is pressed
            if(!indicatorSent) { //should only fire the first time!
                pc.printf("Command on\r\n");
                broadcasterService.sendCommand(cmdIndicatorOn);
                indicatorSent = true; //set true to stop multiple firing
            }
        } else {
            if(!indicatorSent) {
                pc.printf("Command off\r\n");
                broadcasterService.sendCommand(cmdIndicatorOff);
                indicatorSent = true; //set true to stop multiple firing
            }
        }
    }
}