#include "mbed.h"
#include "BLEDevice.h"
#include "BroadcasterService.h"
#include <string>

using namespace std;

BLEDevice ble;
DigitalOut led1(LED1);
InterruptIn indicatorIn(p5);
InterruptIn brakeIn(p2);
Serial pc(USBTX, USBRX);

/*Variable Declarations*/
const static char       DEVICE_NAME[]       = "BLE_CarPeripheral";
uint8_t                 cmdIndicatorOn[8]   = { 0x69,0x6e,0x64,0x69,0xff,0xff,0xff,0xff }; // = I   N   D   I   255 255 255 255
uint8_t                 cmdIndicatorOff[8]  = { 0x69,0x6e,0x64,0x69,0x00,0x00,0x00,0x00 }; // = I   N   D   I   0   0   0   0 
uint8_t                 cmdBrakeOn[8]   = { 0x42, 0x52, 0x41, 0x4b, 0xff,0xff,0xff,0xff }; // = B   R   A   K   255 255 255 255
uint8_t                 cmdBrakeOff[8]  = { 0x42, 0x52, 0x41, 0x4b,0x00,0x00,0x00,0x00 }; // =  B   R   A   K   0   0   0   0 
static const uint16_t uuid16_list[]        = {BroadcasterService::BROADCAST_SERVICE_UUID};

BroadcasterService* broadcasterService;

/** Callback function for ticker */
void blink(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */  
}

/** Callback routine is interrupt activated by a debounced button hit*/
void indicatorSwitchOn(void) {
    pc.printf("Command Indicator on\r\n");
    broadcasterService->sendCommand(cmdIndicatorOn);
}

/** Callback routine is interrupt activated by a debounced button release*/
void indicatorSwitchOff(void) {
    pc.printf("Command Indicator off\r\n");
    broadcasterService->sendCommand(cmdIndicatorOff);
}

/** Callback routine is interrupt activated by a debounced button hit*/
void brakeOn(void) {
    pc.printf("Command Brake on\r\n");
    broadcasterService->sendCommand(cmdBrakeOn);
}

/** Callback routine is interrupt activated by a debounced button release*/
void brakeOff(void) {
    pc.printf("Command Brake off\r\n");
    broadcasterService->sendCommand(cmdBrakeOff);
}

/** Callback function for BLE disconnection */
void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    pc.printf("Disconnected! - start advertising. Handle:%d, Reason:0x%02x\r\n", handle, reason);
    ble.startAdvertising();
}

/** Callback function for BLE connection */
void connectionCallback(Gap::Handle_t handle, Gap::addr_type_t peerAddrType, const Gap::address_t peerAddr, const Gap::ConnectionParams_t *parms)
{
    pc.printf("Connected! - stop advertising. Handle:%d, eType:%d, Addr:%u\r\n", handle, peerAddrType, peerAddr);
    ble.stopAdvertising();
}

/** Callback function for error ticker*/
void raiseGenericError()
{
    pc.printf("Error raised - %u\r\n", BroadcasterService::ERROR_GENERIC);
    broadcasterService->registerError(BroadcasterService::ERROR_GENERIC);
}

int main(void)
{    
    /*Setup switch input*/
    indicatorIn.mode(PullUp);
    brakeIn.mode(PullUp);
    // Delay for initial pullup to take effect
    wait(.01);    
    // Setup Interrupt callback function for a button hit
    indicatorIn.fall(&indicatorSwitchOn);
    indicatorIn.rise(&indicatorSwitchOff);
    
    brakeIn.fall(&brakeOn);
    brakeIn.rise(&brakeOff);
    
    /*Setup Blinky*/
    led1 = 1;
    Ticker t;
    t.attach(&blink, 1);
    
    /*Setup random error timer*/
    int interval = rand() % 100 + 10;
    Ticker errTicker;
    errTicker.attach(&raiseGenericError, interval);

    /*Setup BLE*/
    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onConnection(connectionCallback);

    //create services  
    broadcasterService = new BroadcasterService(ble);

    /*
    * BREDR_NOT_SUPPORTED = BLE only
    * LE_GENERAL_DISCOVERABLE = Device is discoverable at any moment (no time out)
    * ADV_CONNECTABLE_UNDIRECTED = Any central device can connect
    */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    
    ble.setAdvertisingInterval(1600); /* 1s; in multiples of 0.625ms. */
    ble.startAdvertising();

    pc.printf("Advertising node %s\n\r", DEVICE_NAME);

    //loop forever
    while(true) {
        ble.waitForEvent(); // this should return upon any system event (such as an interrupt or a ticker wakeup)                
    }
}