/* mbed Microcontroller Library
 * Copyright (c) 2019 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */

#include "mbed.h"
#include "ShiftReg.h" 
#include "Joystick.h"
#include "N5110.h"
#include "Road.h"
#include "Car.h"
#include "Utils.h"
#include "Vector.h"
#include "Ball.h"
#include "Vector.h"
#include "GameEngine.h"
#include <stdlib.h>     
#include <time.h>       

// objects
// BusOut leds(LED4,LED3,LED2,LED1);

N5110 lcd(p14,p8,p9,p10,p11,p13,p21); 
InterruptIn button_A(p29);
DigitalIn button_B(p28);
DigitalIn button_C(p27);
DigitalIn button_D(p26);
ShiftReg seven_seg;

//                 y   x  
Joystick joystick(p20,p19);

//             B   G   R
//BusOut leds(p22,p23,p24);
//          LSB     MSB

Road road;
Utils utils;
Car car;
Ball ball;
GameEngine game;
Ticker ticker;

// flag - must be volatile as changes within ISR
// g_ prefix makes it easier to distinguish it as global
volatile int g_timer_flag = 0;
volatile int g_buttonA_flag = 0;

#define speed 0.01
#define road_speed 0.016
#define car_speed 0.2

// functions
void init_buttons();
void init();
void render();
void menu();
void instructions();
void pause();
void game_over();
void draw_arrow(int x, int y);
void timer_isr();
void buttonA_isr();
void display_lives(int lives);

int main()
{
    button_A.rise(&buttonA_isr);
    init_buttons();
    int state = 0;  // set inital state
    init();
    // welcome();
    int up_down = 17;
    float number_1 = 0;
    float number_2 = 0;
    int lives = 4;
    int old_collision = 0;
    display_lives(0);
    
    // set-up the ticker so that the ISR it is called every 5 seconds
    ticker.attach(&timer_isr,5);
    
    while(1) {
        lcd.clear();
        Vector2D coord = joystick.get_mapped_coord(); 
        switch(state) {
            
            case 0: // Main Menu
            {
                menu();
                lives = 4;
                display_lives(0);
                if(coord.y >= 0.5) {
                    up_down = 17;
                }
                
                if(coord.y <= -0.5) {
                    up_down = 25;
                }
                
                draw_arrow(2,up_down);
                
                if(g_buttonA_flag) {
                    if(up_down == 17){
                        state = 1;
                        g_buttonA_flag = 0;
                    }
                    if(up_down == 25) {
                        state = 2;
                        g_buttonA_flag = 0;
                    }
                }  
                break;
            }
            
            
            case 1: // Actual Game
            {
                game.update(coord.x);
                
                // check if flag is set i.e. interrupt has occured
                if (g_timer_flag) {
                    g_timer_flag = 0;  // if it has, clear the flag
                    
                    // generate random road or ball
                    // initialize random seed: 
                    srand (time(NULL));
                    
                    number_1 = (rand() % 20 + 1) - 10;
                    number_2 = (static_cast <float> (rand()) / (static_cast <float> (RAND_MAX/10.0))) - 5.0;
                    game.reset_ball();
                }
                
                game.generate_ball(number_1);
                game.generate_road(number_2);
         
                // put the MCU to sleep until an interrupt wakes it up
                sleep();
                render();
                display_lives(lives);
                
                int collision = game.check_collision();
                if(collision == 1 && old_collision == 0) {
                    lives--;
                    if (lives == 0) {
                        state = 4;
                    }
                }
                
                old_collision = collision;
                
                if(g_buttonA_flag) {
                    state = 3;
                    g_buttonA_flag = 0;   
                }
                break;
            }
            
           
            case 2:  // Instructions 
            {
                instructions();
                if(g_buttonA_flag) {
                    state = 0;
                    g_buttonA_flag = 0;
                }
                break;   
            }
                 
                
            case 3: // Pause Menu
            {
                pause();
                if(coord.y >= 0.5) {
                    up_down = 17;    
                }
                if(coord.y <= -0.5) {
                    up_down = 25;
                    }    
                
                draw_arrow(2,up_down);
                if(g_buttonA_flag) {
                    if(up_down == 17){
                        state = 1;
                        g_buttonA_flag = 0;
                    }
                    if(up_down == 25) {
                        state = 0;
                        g_buttonA_flag = 0;
                    } 
                }
                break;
            }
            
            case 4: // Game over
            {
                game_over();
                if(g_buttonA_flag) {
                    state = 0;
                    g_buttonA_flag = 0;
                }
                break;  
            } 
        }
    sleep();
    lcd.refresh();
    }
}

void init_buttons()
{
    // PCB has external pull-down resistors so turn the internal ones off
    // (default for DigitalIn)
    button_A.mode(PullNone);
    button_B.mode(PullNone);
    button_C.mode(PullNone);
    button_D.mode(PullNone);
}


void init() {
    seven_seg.write(0x00); 
    lcd.init();
    joystick.init();
    lcd.setContrast(0.55);
    
    std::vector<Vector2Df> curve_point;
    Vector2Df point = {-20.0,-50.0};
    curve_point.push_back(point);
    
    game.init(0.0,          // road direction
              0.0,          // road inclination
              4,            // ball radius
              curve_point,  // points from ball path
              0,            // car direction
              0,            // car magnitude 
              38.0,         // car start position
              0.0,          // offset
              0.0,          // car_turn
              0.0);         // ball_it
}

void menu() { 
    lcd.printString("MAIN MENU",15,0);
    lcd.printString("Start Game",10,2);
    lcd.printString("Instructions",10,3); 
    lcd.printString("Press A",20,5); 
}

void instructions() {  
    lcd.printString("INSTRUCTIONS",6,0);
    lcd.printString("Use joystick",6,1);
    lcd.printString("to move",21,2);
    lcd.printString("Press A to",13,4);
    lcd.printString("pause or exit",4,5);
    display_lives(0);
}

void pause() { 
    lcd.printString("PAUSE",27,0);
    lcd.printString("Back to game",10,2);
    lcd.printString("Back to menu",10,3); 
}

void game_over() {
    lcd.printString("GAME OVER",16,1); 
    lcd.printString("Don't collide!",1,2); 
    lcd.printString("Press A",20,5); 
}

void render() { 
    game.draw(lcd, utils);
}

void draw_arrow(int x, int y) {
    const int arrow[5][7] = {
        { 0, 0, 0, 0, 1, 0, 0 },
        { 1, 1, 1, 1, 1, 1, 0 },
        { 1, 1, 1, 1, 1, 1, 1 },
        { 1, 1, 1, 1, 1, 1, 0 },
        { 0, 0, 0, 0, 1, 0, 0 },
    };  
    lcd.drawSprite(x,y,5,7,(int *)arrow);
}

// time-triggered interrupt
void timer_isr() {
    g_timer_flag = 1;   // set flag in ISR
}

// Button A event-triggered interrupt
void buttonA_isr() {
    g_buttonA_flag = 1;   // set flag in ISR     
}

void display_lives(int lives) {
    if (lives == 4) {
        lcd.printString("Lives: 4",0,0);
        seven_seg.write(0x66);
    } else if (lives == 3) {
        lcd.printString("Lives: 3",0,0);
        seven_seg.write(0x4F);
    } else if (lives == 2) {
        lcd.printString("Lives: 2",0,0);
        seven_seg.write(0x5B);
    } else if (lives == 1) {
        lcd.printString("Lives: 1",0,0);
        seven_seg.write(0x06);
    } else {
        seven_seg.write(0x3F);
    }
}