//remove repetition
#ifndef MBED_JOYSTICK_H
#define MBED_JOYSTICK_H

//required to use mbed functions
#include "mbed.h"

/** Struct: joyhv
 *
 * Used for holding a horizontal and vertical position as doubles
 */
struct joyhv {
    double h;
    double v;
};

/** Class: Joystick
 *
 * Used for reading from an analog joystick
 *
 * Example:
 *
 * > #include "mbed.h"
 *
 * > Joystick joy(p20, p19, p18);
 */
 
class Joystick {
public:
    /** Constructor: Joystick
     *
     * Variables: 
     * b - DigitalIn pin for button
     * h - AnalogIn pin for horizontal 
     * v - AnalogIn pin for vertical 
     */ 
    Joystick(PinName b, PinName h, PinName v);
    
    /** Function: read
     * Read the joystick position, represented as a joyhv value - h and v are doubles in the range [0.0, 1.0]
     *
     * Variables:
     *  returns - A structure of two double values representing the position of the joystick,
     *            measured as a percentage vertically (joyhv.v) or horizontally (joyhv.h)
     */
    joyhv read();

    /** Function: getV
     * Read the joystick's vertical position, represented as a double value in the range [0.0, 1.0]
     *
     * Variables:
     *  returns - A double values representing the vertical position of the joystick,
     *            measured as a percentage  
     */
    double getV();
    
    /** Function: getH
     * Read the joystick's horizontal position, represented as a double value in the range [0.0, 1.0]
     *
     * Variables:
     *  returns - A double values representing the horizontal position of the joystick,
     *            measured as a percentage  
     */
    double getH();
    
    /** Function: rise
     *  Attach a function to call when a rising edge occurs on the button input
     *
     * Variables:
     *  fptr - A pointer to a void function, or 0 to set as none
     */
    void rise (void (*fptr)(void));
    
    /** Function: fall
     *  Attach a function to call when a falling edge occurs on the button input
     *
     * Variables:
     *  fptr - A pointer to a void function, or 0 to set as none
     */
    void fall (void (*fptr)(void));
    
    /** Function: operator joyhv
     *  An operator shorthand for <read()>
     *
     * The joyhv() operator can be used as a shorthand for <read()> to simplify common code sequences
     *
     */
    operator joyhv ();
    
    joyhv scale(joyhv read);
    joyhv filter(joyhv read, double factor);

    
private:
    InterruptIn _b;
    AnalogIn _h;
    AnalogIn _v;
};


#endif