
#include "mbed.h"

#include "stm32l475e_iot01_accelero.h"

#include <stdio.h>
#include <errno.h>

    // Block devices
#if COMPONENT_SPIF
#include "SPIFBlockDevice.h"
#endif

#if COMPONENT_DATAFLASH
#include "DataFlashBlockDevice.h"
#endif

#if COMPONENT_SD
#include "SDBlockDevice.h"
#endif

#include "HeapBlockDevice.h"

    // File systems
#include "LittleFileSystem.h"
#include "FATFileSystem.h"
#include "nvstore.h"

    // Physical block device, can be any device that supports the BlockDevice API
/*SPIFBlockDevice bd(
 MBED_CONF_SPIF_DRIVER_SPI_MOSI,
 MBED_CONF_SPIF_DRIVER_SPI_MISO,
 MBED_CONF_SPIF_DRIVER_SPI_CLK,
 MBED_CONF_SPIF_DRIVER_SPI_CS);*/

#define X_VALUE pDataXYZ[0]
#define Y_VALUE pDataXYZ[1]
#define Z_VALUE pDataXYZ[2]

#define HORIZONTAL_INDEX 0
#define LONG_VERTICAL_INDEX 1
#define SHORT_VERTICAL_INDEX 2

#define BLOCK_SIZE 512
HeapBlockDevice bd(16384, BLOCK_SIZE);

    // File system declaration
LittleFileSystem fs("fs");

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);

Ticker tTicker;
Thread t;

uint16_t horizontalKey = 10;
uint16_t longVerticalKey = 11;
uint16_t shortVerticalKey = 12;

uint32_t horizontalCount;
uint32_t longVerticalCount;
uint32_t shortVerticalCount;

static FILE *f;
volatile int countedPositionsd = 0;
EventQueue queue(32 * EVENTS_EVENT_SIZE);

InterruptIn button(USER_BUTTON);

int max3(int a, int b, int c) {
    if (a > b) {
        return a > c ? HORIZONTAL_INDEX : SHORT_VERTICAL_INDEX;
    }
    return b > c ? LONG_VERTICAL_INDEX : SHORT_VERTICAL_INDEX;
}

void turnOffLeds() {
    led1 = 0;
    led2 = 0;
    led3 = 0;
}

void turnOnLedAndUpdateStore(DigitalOut * led, uint16_t key, uint32_t * currentCount) {
    *led = 1;
    NVStore &nvstore = NVStore::get_instance();
    *currentCount = *currentCount + 1;
    nvstore.set(key, sizeof(uint32_t), currentCount);
}

void toggleLed() {
    int16_t occurencies[3] = {0};
    fflush(stdout);
    fflush(f);
    
    fseek(f, 0, SEEK_SET);
    
    printf("Start reading\n");
    
    int number;
    while (!feof(f)) {
        fscanf(f, "%d", &number);
        if (number == HORIZONTAL_INDEX || number == LONG_VERTICAL_INDEX || number == SHORT_VERTICAL_INDEX) {
            occurencies[number] = occurencies[number] + 1;
        }
    }
    
    printf("%d %d %d\n", occurencies[HORIZONTAL_INDEX], occurencies[LONG_VERTICAL_INDEX], occurencies[SHORT_VERTICAL_INDEX]);
    
    int mostOccurentValue = max3(occurencies[0], occurencies[1], occurencies[2]);
    
    printf("Max %d\n", mostOccurentValue);
    
    turnOffLeds();
    
    if (mostOccurentValue == HORIZONTAL_INDEX) {
        turnOnLedAndUpdateStore(&led1, horizontalKey, &horizontalCount);
    } else if (mostOccurentValue == LONG_VERTICAL_INDEX) {
        turnOnLedAndUpdateStore(&led2, longVerticalKey, &longVerticalCount);
    } else if (mostOccurentValue == SHORT_VERTICAL_INDEX) {
        turnOnLedAndUpdateStore(&led3, shortVerticalKey, &shortVerticalCount);
    }
    
    fflush(stdout);
    int err = fclose(f);
    printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
    if (err < 0) {
        error("error: %s (%d)\n", strerror(err), -err);
    }
    err = fs.unmount();
    printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
    if (err < 0) {
        error("error: %s (%d)\n", strerror(-err), err);
    }
}

bool isInRange100(int val) {
    return val < 100 && val > -100;
}

bool isInRange900_1000(int val) {
    return (val < 1050 && val > 950) || (val < -950 && val > -1050);
}

void savePosition() {
    int16_t pDataXYZ[3] = {0};
    BSP_ACCELERO_AccGetXYZ(pDataXYZ);
    if (isInRange100(X_VALUE) && isInRange100(Y_VALUE) && isInRange900_1000(Z_VALUE)) {
        fprintf(f, "%d\n", HORIZONTAL_INDEX);
    } else if (isInRange100(X_VALUE) && isInRange900_1000(Y_VALUE) && isInRange100(Z_VALUE)) {
        fprintf(f, "%d\n", LONG_VERTICAL_INDEX);
    } else if (isInRange900_1000(X_VALUE) && isInRange100(Y_VALUE) && isInRange100(Z_VALUE)) {
        fprintf(f, "%d\n", SHORT_VERTICAL_INDEX);
    }
    fflush(f);
    fflush(stdout);
}


void tickerBlock() {
    queue.call(savePosition);
    countedPositionsd++;
    if (countedPositionsd == 1000) {
        tTicker.detach();
        queue.call(toggleLed);
        countedPositionsd = 0;
    }
}

int main(){
    t.start(callback(&queue, &EventQueue::dispatch_forever));
    BSP_ACCELERO_Init();
    
    NVStore &nvstore = NVStore::get_instance();
    uint16_t actual_len_bytes = 0;
    int rc;
    
    rc = nvstore.init();
    
    rc = nvstore.get(horizontalKey, sizeof(horizontalCount), &horizontalCount, actual_len_bytes);
    if (rc == NVSTORE_NOT_FOUND) {
        horizontalCount = 0;
        shortVerticalCount = 0;
        longVerticalCount = 0;
        nvstore.set(horizontalKey, sizeof(horizontalCount), &horizontalCount);
        nvstore.set(shortVerticalKey, sizeof(shortVerticalCount), &shortVerticalCount);
        nvstore.set(longVerticalKey, sizeof(longVerticalCount), &longVerticalCount);
    } else {
        nvstore.get(shortVerticalKey, sizeof(shortVerticalCount), &shortVerticalCount, actual_len_bytes);
        nvstore.get(longVerticalKey, sizeof(longVerticalCount), &longVerticalCount, actual_len_bytes);
    }
    
    printf("Led 1: %d\nLed 2: %d\nLed 3: %d\n", horizontalCount, longVerticalCount, shortVerticalCount);
        // Try to mount the filesystem
    printf("Mounting the filesystem... ");
    fflush(stdout);
    int err = fs.mount(&bd);
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
            // Reformat if we can't mount the filesystem
            // this should only happen on the first boot
        printf("No filesystem found, formatting... ");
        fflush(stdout);
        err = fs.reformat(&bd);
        printf("%s\n", (err ? "Fail :(" : "OK"));
        if (err) {
            error("error: %s (%d)\n", strerror(-err), err);
        }
    }
    
        // Open the numbers file
    printf("Opening \"/fs/numbers.txt\"... ");
    fflush(stdout);
    f = fopen("/fs/numbers.txt", "r+");
    printf("%s\n", (!f ? "Fail :(" : "OK"));
    if (!f) {
            // Create the numbers file if it doesn't exist
        printf("No file found, creating a new file... ");
        fflush(stdout);
        f = fopen("/fs/numbers.txt", "w+");
        printf("%s\n", (!f ? "Fail :(" : "OK"));
        if (!f) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
    }
    
    tTicker.attach(&tickerBlock, 0.01);
}
