#include "mbed.h"

DigitalIn sw1(PC_14);
DigitalIn sw2(PH_0);
DigitalIn sw3(PC_0);
DigitalIn sw4(PC_1);
DigitalIn sw5(PC_2);
DigitalIn sw6(PC_3);

DigitalOut led1(PC_13);
DigitalOut led2(PC_15);
DigitalOut led3(PH_1);
DigitalOut led4(PC_4);
DigitalOut lcd_bl(PC_5);

DigitalOut gps_en(PA_11);

Serial ser_daplink(PB_6, PB_7);
Serial ser_external(PA_2, PA_3);
Serial ser_gps(PC_6, PC_7);

I2C i2c_lcd(PB_9, PB_8);

static void lcd_init();
static int lcd_puts(uint8_t *str, uint8_t start_index);
void lcd_clear();

int main() {
    uint8_t mode = 0xff;
    
    ser_daplink.baud(9600);
    ser_external.baud(9600);
    ser_gps.baud(9600);
    
    gps_en = 1;
    
    lcd_init();
    lcd_puts((uint8_t*)"Serial", 0);
    
    while(1) {
        led1 = sw1;
        led2 = sw2;
        led3 = sw3;
        led4 = sw4;
        
        int val = sw5 + (sw6 << 1);
        
        if (val != mode) {
            char mode_str[17];
            sprintf(mode_str, "Mode: %d", val);
            lcd_puts((uint8_t *)mode_str, 16);
            mode = val;
        }
        
        switch (val) {
            case 0:
                lcd_bl = 0;
                break;
            case 1:
                // DAPLink loopback
                if (ser_daplink.readable()) {
                    ser_daplink.putc(ser_daplink.getc());
                }
                lcd_bl = 1;
            case 2:
                // DAPLink - Extended relay
                if (ser_daplink.readable()) {
                    ser_external.putc(ser_daplink.getc());
                }
                if (ser_external.readable()) {
                    ser_daplink.putc(ser_external.getc());
                }
                lcd_bl = 1;
                break;
            case 3:
                // DAPLink - GPS module relay
                if (ser_daplink.readable()) {
                    ser_gps.putc(ser_daplink.getc());
                }
                if (ser_gps.readable()) {
                    ser_daplink.putc(ser_gps.getc());
                }
                lcd_bl = 1;
                break;
            default:
                break;
        }
    }
}

//#define LCD_ADR (0x3e)
#define LCD_ADR (0x7c)
static int lcd_write_cmd(uint8_t cmd)
{
    uint8_t dat[2];
    dat[0] = 0x00;
    dat[1] = cmd;
    return i2c_lcd.write(LCD_ADR, (char*)dat, 2, false);
}

static int lcd_write_char(uint8_t chr)
{
    uint8_t dat[2];
    dat[0] = 0x40;
    dat[1] = chr;
    return i2c_lcd.write(LCD_ADR, (char*)dat, 2, false);
}

static bool lcd_loc(uint8_t col, uint8_t row)
{
    uint8_t addr = 0x80;
    if ((col < 16) && (row < 2)) {
        addr |= col;
        if (row == 1) {
            addr |= 0x40;
        }
        lcd_write_cmd(addr);    //DDRAMのアドレスを設定
        return true;
    }

    return false;
}

void lcd_clear(void)
{
    lcd_write_cmd(0x01);
    wait_us(10000);
}

static void lcd_init()
{
    //LCD initialize
    wait_us(50000);
    lcd_write_cmd(0x38);
    wait_us(30);
    lcd_write_cmd(0x02);
    wait_us(30);
    lcd_write_cmd(0x39);
    wait_us(30);
    lcd_write_cmd(0x14);
    wait_us(30);
    lcd_write_cmd(0x7f);
    wait_us(30);
    lcd_write_cmd(0x56);
    wait_us(30);
    lcd_write_cmd(0x6a);
    
    wait_us(200000);
    
    lcd_write_cmd(0x0c);
    wait_us(30);
    lcd_write_cmd(0x01);
    wait_us(30);
    lcd_write_cmd(0x04);
    wait_us(30);
}

static int lcd_puts(uint8_t *str, uint8_t start_index)
{
    int i;
    uint8_t addr = 0x80;
    
    if (start_index >= 32) {
        return -1;
    }
    
    //表示開始アドレス
    if (start_index < 16) {
        //1行目
        addr |= start_index;
    } else {
        //2行目
        addr |= (0x40 | (start_index - 16));
    }
    lcd_write_cmd(addr);
    wait_us(30);
    
    for (i = 0; i < 32; i++) {
        if (str[i] == '\0') {
            break;
        }
        
        if ((i + start_index) == 16) {
            lcd_write_cmd(0xc0);    //アドレスを2行目先頭に
            wait_us(30);
        } else if ((i + start_index) == 32) {
            lcd_write_cmd(0x80);    //アドレスを1行目先頭に
            wait_us(30);
        }
        lcd_write_char(str[i]);
        wait_us(30);
    }
    
    return i;
}
