/**
 *  TRP105F_Spline.h,.cpp
 *
 *  Author: aktk, aktk.j.uec@gmail.com
 *  Tokyo, Japan.
 *
 *  -   This library is for providing a distance from TRP105F, photo-reflect proximity sensor,
 *  to whatever reflect the flash.
 *  -   Deistance is derived as a voltage signal of TRP105F.
 *  -   An object of TRPFS class has an array of sets of
 *  distance(16 bit-value) and voltage(16bit-value) at the distance.
 *  -   The range of distances that an array has in sets is [LIDX:RIDX] by 1.
 *  -   an array is derived from a cubic spline curve model.
 *  -   In order to derive spline curve, some value sets should be got
 *  at the first calibration.
 *
 *  LOG:
 *  ver.1   2015/10/19 - 2015/10/22
 *  ver.2   2015/10/22 -
 *      Distance data type has become unsigned short from int.
 *      Distance data range : [2:20] by 1 -> [0:1024] by 1.
 *  ver.2.1 2016/02/12 - 2016/02/16
 *      Distance data range : [0:1024] -> [0:255].
 *  ver.3.0 2016/05/23 - 2016/06/08
 *      Some Functions modified a lot. Even same name function might become uncompatible!!!!!
 *      (in order to make it be extensivly)
 *      Following are Implemented: 
 *          -Functions relating to save/load data with serial
 *          -Some New Members: _ai, or so on.
 *          -enum UseType
 *      
 */
 
#ifndef TRP105F_SPLINE_H
#define TRP105F_SPLINE_H


#include "mbed.h"
#include "CODE_SELECTIVE.h"


//  Set of voltage-distance
typedef struct {
    unsigned short x; //  distance
    unsigned short y; //  voltage
} VDset;

//  Type of modality to input data for calibration
enum UseType {
    AsDEBUG,
    AsMODULE
} ;

//
//  TRP105FS Class for get distance from voltage
//
class TRP105FS
{
public:
    //  Constraction
    TRP105FS();
    TRP105FS(unsigned short);
    TRP105FS(unsigned short, UseType);
    TRP105FS(unsigned short, PinName);
    TRP105FS(unsigned short, UseType, PinName);
    //TRP105FS(unsigned int arg_num, DataInType arg_dit, unsigned int channel);
    //  Destraction
    ~TRP105FS();
    //  Functions
    unsigned short getDistance(unsigned short); // alias of getX
    unsigned short getDistance();   // get voltage from PinName and getX()
    unsigned short getVoltage();    // get voltage
    unsigned short getX(unsigned short);    //  the fuction to get distance.
    unsigned short getY(unsigned short);    //  get y-spline_model value of x(argument)
    unsigned short getSampleNum();
    void    calibrateSensor();
    //void    calibrateSensor(VDset* arg_set, unsigned int arg_num);
    void    calibrate();
    void    setSample(unsigned short,unsigned short);
#ifdef HAS_LOCAL_FILE_SYSTEM
    void    printOutData(const char* filename);
    void    saveSetting();
    void    loadSetting();
    void    saveSetting(const char* filename);
    void    loadSetting(const char* filename);
    //void    saveSetting_fromSirial(const char* filename, Serial* com);
    //void    loadSetting_intoSerial(const char* filename, Serial* com);
#endif
    void    saveSetting_intoSerial(Serial *com);
    void    loadSetting_fromSerial(Serial *com);
#ifdef HAS_COM_TO_CONSOLE
    void    printThresholds();
#endif

private:
    //
    //  Defining Constants
    //
    enum {
        _LIDX = 0,
        _RIDX = 255,
        _ENUM = _RIDX - _LIDX + 1
    };
    //
    //  Variables
    //
    //  Debug or Module
    UseType         _useType;
    //  For data sampling and making spline model
    unsigned short  _Sample_Num;    // the number of samples for derive spline
    unsigned short  _snum;          //  for counting set sample data by _setSample(us,us);
    VDset*          _Sample_Set;
    double*         _u_spline;
    //  For comvert voltage -> physical quantity e.g. distance
    VDset           _Set[_ENUM];
    unsigned short  _Threshold[_ENUM]; //_Threshold[18] is not used virtually.
    //  For get voltage
    AnalogIn        _ai;
    //
    //
    //  For calibration
    //
#ifdef HAS_COM_TO_CONSOLE
    void    _sampleData();
#endif
    void    _setSample(unsigned short,unsigned short);
    void    _setSamples(VDset* arg_set, unsigned int arg_num);
    unsigned short _getSplineYof(double arg_x);
    void    _makeSpline();  //  Cubic spline
    //
    //  For get distance
    //
    int     _getNearest(int, int, unsigned short);
    //
    //  For debug
    //
#ifdef HAS_LOCAL_FILE_SYSTEM
    void    _printOutData(unsigned short    *arg, int num, char* name);
    void    _printOutData(VDset             *arg, int num, char* name);
    void    _printOutData(double            *arg, int num, char* name);
#endif
};

//inline implemeted

inline unsigned short TRP105FS::getDistance(unsigned short arg_y)
{
    return getX(arg_y);
}

inline unsigned short TRP105FS::getDistance()
{
    return getX(_ai.read_u16());
}

inline unsigned short TRP105FS::getVoltage()
{
    return _ai.read_u16();
}

#endif