/** SwArr16MOSFET class header file

 *  \file   SwArr16MOSFET.h
 *  \author Akifumi Takahashi
 *  \date   2019/jan/28 ver.1.FirstImplement
 *  \version 1.0.2019.jan.FirstImpelementation
 *  \version 1.1.2019.july.TentativelyCompleted
 */
 
#ifndef PMRC_16CHANNEL_H
#define PMRC_16CHANNEL_H
#include "mbed.h"

/** SwArr16MOSFET
 *  
 *  \par    About SwArr16MOSFET class
 *  - A Program with which to control MOSFET Half Bridges array Circuit
 *  for 16 channels Electric Stimulation.
 *  - P/N type MOSFETs are controlled by Shift Resisters. So this lib serves 
 *  shift resister's utility.
 *  - A Half Bridge Circuit has P type MOSFET and N type one and they have 
 *  characteristics opposit each other, so it is required to modify output of
 *  Shift Resistors from the program to use Photo MOS Relay Circuit.
 *
 *  \par    Image of MOSFET Half Bridges Array
 *  -   32 Half Briges Array IN 
 *  -   16 channel OUT 
 *  
 *  \verbatim
 *   +--i      o----+----i      o--+
 *   |  [FET01-P]   |    [FET01-N] |
 *   |         (ch1(B)out)         |
 *   |                             |
 *   +--i      o----+----i      o--+
 *   |  [FET02-P]   |    [FET02-N] |
 *   |         (ch2(R)out)         |
 *   |                             |
 *   .              .              .
 *   .              .              .
 *   .              .              .
 *  Vpp                           GND
 *  \endverbatim
 *
 *  \par Info. Shit-Resister circuit board
 *  -   nOE is connected to GND
 *
 *  \verbatim
 *  in
 *  ->[FET01-P]->[FET01-N]->[FET02-P]->[FET02-N]
 *  ->[FET03-P]->[FET03-N]->[FET04-P]->[FET04-N]
 *  ->[FET05-P]->[FET05-N]->[FET06-P]->[FET06-N]
 *  ->[FET07-P]->[FET07-N]->[FET08-P]->[FET08-N]
 *  ->[FET09-P]->[FET09-N]->[FET10-P]->[FET10-N]
 *  ->[FET11-P]->[FET11-N]->[FET12-P]->[FET12-N]
 *  ->[FET13-P]->[FET13-N]->[FET14-P]->[FET14-N]
 *  ->[FET15-P]->[FET15-N]->[FET16-P]->[FET16-N]
 *  ->out
 *  \endverbatim
 *
 *  \version ver.1.FirstImplement; derived from PMRC16ch
 */
class SwArr16MOSFET
{
public:
    //
    //  Constructors
    //  ------------------------------------------------------------------------
    
    /// Constructor default
    SwArr16MOSFET();
    
    /// Constructor with setting num of channels
    SwArr16MOSFET(
        uint8_t arg_num_ch  ///<[in] number of channels to use
    );
    
    /// Construxtor with full configuration
    SwArr16MOSFET(
        uint8_t arg_num_ch, ///<[in] number of channels to use
        PinName arg_SER,    ///<[in] pin to insert data
        PinName arg_SCK,    ///<[in] pin of shift registor clock
        PinName arg_RCK,    ///<[in] pin of storage registor (buffer) clock
        PinName arg_CLR,    ///<[in] pin to clear the shift registor
        PinName arg_MD_OE   ///<[in] pin to switch MOSFET Driver ON/OFF
    );

    //
    //  Const.
    //  ------------------------------------------------------------------------
    
    /** Const. of channel state. 
     *  
     *  To be used to make signal to shift resister.
     */
    enum State {
        ALLGND = /*FET01-P*/0xFFFFFFFF/*FET16-N*/,
        CH01Lo = /*FET01-P*/0b11000000000000000000000000000000/*FET16-N*/,
        CH02Lo = /*FET01-P*/0b00110000000000000000000000000000/*FET16-N*/,
        CH03Lo = /*FET01-P*/0b00001100000000000000000000000000/*FET16-N*/,
        CH04Lo = /*FET01-P*/0b00000011000000000000000000000000/*FET16-N*/,
        CH05Lo = /*FET01-P*/0b00000000110000000000000000000000/*FET16-N*/,
        CH06Lo = /*FET01-P*/0b00000000001100000000000000000000/*FET16-N*/,
        CH07Lo = /*FET01-P*/0b00000000000011000000000000000000/*FET16-N*/,
        CH08Lo = /*FET01-P*/0b00000000000000110000000000000000/*FET16-N*/,
        CH09Lo = /*FET01-P*/0b00000000000000001100000000000000/*FET16-N*/,
        CH10Lo = /*FET01-P*/0b00000000000000000011000000000000/*FET16-N*/,
        CH11Lo = /*FET01-P*/0b00000000000000000000110000000000/*FET16-N*/,
        CH12Lo = /*FET01-P*/0b00000000000000000000001100000000/*FET16-N*/,
        CH13Lo = /*FET01-P*/0b00000000000000000000000011000000/*FET16-N*/,
        CH14Lo = /*FET01-P*/0b00000000000000000000000000110000/*FET16-N*/,
        CH15Lo = /*FET01-P*/0b00000000000000000000000000001100/*FET16-N*/,
        CH16Lo = /*FET01-P*/0b00000000000000000000000000000011/*FET16-N*/,
        ALLHiZ = /*FET01-P*/0x55555555/*FET16-N*/
    };
    
    /** Polarity
     *
     *  If Anodic the signal remains as is, Cathodic it gets inversed.
     */
    enum Polarity {Anodic = 1, Cathodic = 0};
    
    /** Stimulation Mode
     *
     *  If ONE_VS_THEOTHERS one channel gets one to stimulate and the other are references.
     *  If TWIN_ELECTRODES, two channel get pair to stimulate and the others are disconnected.
     */
    enum StimMode {ONE_VS_THEOTHERS, TWIN_ELECTRODES};
    
    //
    //  Function to prepare channels
    //  ------------------------------------------------------------------------
    
    /** Make the all channels GND. */
    void allGround();
    
    /** Make the all channels HiZ */
    void allHiZ();
    
    
    void repulse();
    
    /** set or change the polarity of stimulation */
    void setPol(Polarity pol);  //inline definition
    
    /** Set a channel to stimulate and the other reference */
    void setOvsO(char ch);
    
    /** Choose a pair of channels with which stimulating in mode of TWIN_ELECTRODES. */
    void setTwin(char stim_ch, char ref_ch);
    
    void setTrio(char stim_ch, char ref_ch1, char ref_ch2);
    
    void setVccCh(char ch);
    
    void setGndCh(char ch);
    
    /**  Initialization
     *
     *  This is called in the constructor.
     */
    void init();
    
    //
    //  Function to get prameter
    //
    
    /** Returning the State code which indicate a state of channels */
    uint32_t getState();        //inline definition
    
    /** Returning the Polarity */
    bool getPol();              //inline definition

private:
    //  The number of channels to use
    const uint8_t m_num_ch;
    static const uint32_t m_statearray[];
    
    //
    //  Var of signals to controll PRM circuit
    //  ------------------------------------------------------------------------
    DigitalOut m_SER;   //  Serial input to shift data in a shift-resister
    DigitalOut m_SCK;   //  Shift-resister's clock
    DigitalOut m_RCK;   //  FF's clock
    DigitalOut m_CLR;   //  To use when you want to clear shift-resister
    DigitalOut m_MD_OE; //  Circuit's Outputs Enable
    
    //
    //  Var of state
    //  ------------------------------------------------------------------------
    //  Position of stimulation bits
    //  0: no stimbits, 1~m_num_ch: channel of stimulation
    uint32_t m_pos_stim;
    uint32_t m_state;
    Polarity m_POL;
    StimMode m_mode;
    
    //
    //  Function to control buffer of shift-resister
    //  ------------------------------------------------------------------------
    //  shift the bits arbitary times
    void shiftby(int times);
    //  Function with which to rapidly set it all-ground
    void sweep();
    //  set the simulation bits of 01
    void setStimbits();
    //  set all bits arbitarily
    void setBits(const uint32_t bits);
    
    //
    //  Clock Functions
    //  ------------------------------------------------------------------------
    //update the clock of shift-resister
    void update();
    //  output the data by uploading them to FF Resitors
    void upload();
    
    //
    //  Output-Enable Functions 
    //  ------------------------------------------------------------------------
    void disable_HiZ();
    void enable();
};
inline void SwArr16MOSFET::setPol(Polarity arg_pol)
{
    m_POL = arg_pol;
}
inline uint32_t SwArr16MOSFET::getState()
{
    return m_state;
}
inline bool SwArr16MOSFET::getPol()
{
    //return false if cathodic
    //return true  if anodic
    return static_cast<bool>(m_POL);
}
#endif