#ifndef SERIAL_INPUT_REACTION_HANDLER_H
#define SERIAL_INPUT_REACTION_HANDLER_H

#include "mbed.h"
#include <string>

/** \Class SerialInputReactionHandler
 *
 *  Char string signal handler. 
 *  This require a serial object,
 *  receiving signal from the serial object,
 *  executing callback which can receive serial as the argument
 */
class SerialInputReactionHandler
{
public:
    /// Constant representing input mode. 
    enum InputMode {KB_SINGLE_INPUT = 0, KB_TILL_ENTER, NUMBEROFMODES};

    /// Constructor
    SerialInputReactionHandler(
        /// Register a callback supposed called when a command is ordered
        Callback<void * (char const * const)> arg_callback_onCommand = &echoCommand,
        /// Register a callback supposed called just before a command be conducted
        Callback<void (char const * const)> arg_commonPre_callback_onCommand = &doNothing,
        /// Register a callback supposed called just after a command be conducted
        Callback<void (char const * const, void *)> arg_commonPost_callback_onCommand = &doNothing
    );
    
    /// Register a callback supposed called when a command is ordered 
    void attach(Callback<void * (char const * const)> arg_callback_onCommand);
    
    /// Register a callback supposed called just before a command be conducted
    void attach_PreProc(Callback<void (char const * const)> arg_commonPre_callback_onCommand);
    
    /// Register a callback supposed called just after a command be conducted
    void attach_PostProc(Callback<void (char const * const, void *)> arg_commonPost_callback_onCommand);
    
    /// Register a serial object and start listening signal from it
    void startReception(Serial * arg_serial_socket, InputMode arg_mode = KB_SINGLE_INPUT);
    
    /// stop listening
    void quit();

    /// Change input mode
    void changeMode(InputMode arg_mode);

private:
    void listenKBSingleInput();
    void listenKBTillEnter();
    void discardBuffer();

    void (SerialInputReactionHandler::*funcIfInput[NUMBEROFMODES])();

    /** Wrapper function that executes commonProcedure() and a function
     *  listed in funcIfInput[]()
     *
     *  This function is supposed to be used as callback attached to such like
     *  Serial.attach(), so that it is called evrey a key input thry keyboard
     *  or a byte written thru communication with like PCs.
     */
    void sig_bind();
    
    void callback_onCommand(char const * const);
    Callback<void * (char const * const)>       m_callback_onCommand;
    Callback<void (char const * const)>         m_Pre_callback_onCommand;
    Callback<void (char const * const, void *)> m_Post_callback_onCommand;
    static void * echoCommand (char const * const);
    template <typename T>
    static void doNothing(T arg){}
    template <typename T1, typename T2>
    static void doNothing(T1 arg1, T2 arg){}
    //static void doNothing(char const * const);
    //static void doNothing(void *);

    Serial * m_serial_socket;

    InputMode m_input_mode;
    uint8_t m_buffer_c;
    string  m_buffer_s;
    
    bool isArrowSymbol();
};

inline void * SerialInputReactionHandler::echoCommand (char const * const arg_str)
{
    printf("%s", arg_str);
    return NULL;
}
inline void SerialInputReactionHandler::discardBuffer()
{
    m_buffer_c = '\0';
    m_buffer_s = "";
}


inline bool isUpperCase (const char arg_c)
{
    return ('A' <= arg_c && arg_c <= 'Z');
}

inline bool isLowerCase (const char arg_c)
{
    return ('a' <= arg_c && arg_c <= 'z');
}

inline bool isSymbol (const char arg_c)
{
    return ((0x20 <= arg_c && arg_c <= 0x40)
            ||  (0x5B <= arg_c && arg_c <= 0x60)
            ||  (0x7B <= arg_c && arg_c <= 0x7E));
}

#endif
