/**     Pulse Fenerator header file
 *
 *  \file   PulseFenerator.h
 *  \author Akifumi Takahashi
 *  \date   2018.nov.08-
 *  \version    1.0.2018.nov
 *  \version    1.1.2019.july
 */
#ifndef PULSEGENERATOR_H
#define PULSEGENERATOR_H
#include "mbed.h"

/**     \class PulseGenerator
 *      A class which generate us-order Pulse wave
 *
 *  This generates continual pulses which can be used for such as clock
 *  for another continual pulses having same pulse width and frequency and
 *  another pulse hight.
 *
 *  start________|--|________|--|________|--|________|--|_____...stop
 */
 
class PulseGenerator
{
public://-----------------------------------------------------------------------
    /**    Default constructor
     *
     *  As default, the parameters of pulse form are set as followng.
     * -   pulse width  = 200   [us]
     * -   pulse period = 5000  [us] (refering to freq = 200 [Hz])
     */
    PulseGenerator();

    /**     Function to set pulse width and culc period from given frequency
     *
     *  This function set a value m_clockduration, m_rt, and m_ft.
     *  The clock is culcurated from pulse width and period (= 1/frequency)
     *  using Eucledean Algorithm.
     */
    int8_t setWaveform(
        const uint16_t arg_pulseWidth,///< required in [us]
        const uint16_t arg_pulseFreq  ///< required in [Hz] bigger than 15 Hz
    );

    /**     Function to start/resume generating Pulse wave
     *
     *  This function culculates a base clock freq (e.g. pulse periodand) 
     *  and a pulse rising/falling timing in the clock with which to generate 
     *  pulse wave. 
     *  And then exe TICKER.attach().
     */
    void startPulseWave();

    /**     Function to stop/pose generating pulse wave
     *
     *  This function exe TICKER.detach().
     */
    void stopPulseWave();

    /**     Function to refer whther the pulse wave is H xor L
     *  \retval (bool) H between pulse rising and then falling
     *  \retval (bool) L between pulse falling and then rising
     */
    bool getState();
    
private://----------------------------------------------------------------------
    uint16_t m_width;       ///< Defalt: 200 [us]; Set in setWaveform()
    uint16_t m_period;      ///< culcurated from frequency given with setWaveform()
    bool     m_wavestatus;
    uint16_t m_clockperiod; ///< base clock's period
    uint16_t m_rt;   ///< Pulse rising timing in the clock (What times clock counting is required till a pulse rises)
    uint16_t m_ft;   ///< Pulse falling timing in the clock (What times clock counting is required till a pulse falls)
    Ticker   m_ticker;

    ///     Pulse Generate Hundler
    void generatePulseWave();
};

//  
//  Inline Implementation
//  ----------------------------------------------------------------------------
inline bool PulseGenerator::getState()
{
    return m_wavestatus;
}

//  
//  Auxiliry Function
//  ----------------------------------------------------------------------------
/**     Culcurate Grate Common Divisor by Eukleides' Algorithm
 *
 *  \param[in]  a value included in Natural number excluding 0
 *  \param[in]  a value included in Natural number excluding 0
 *  \retval GCD
 */
uint16_t GCD(uint16_t, uint16_t);

/**     Culcurate Period in [us] from frequency in [Hz]
 *
 *  \param[in]  Frequency [Hz]
 *  \retval Period [us]
 */
uint16_t Period_us(const uint16_t arg_freq);
#endif