/** PMRC16 class header file
 *  \file   PMRC16ch.h
 *  \author Akifumi Takahashi
 *  \date   2016/dec/11-
 *  -   ver.1;      8 channel
 *  \date   2018/jun/01-
 *  -   ver.16.1;   16 channel
 *  \date   2018/jun/28-
 *  -   ver.16.2;   major change
 */
#ifndef PMRC_16CHANNEL_H
#define PMRC_16CHANNEL_H
#include "mbed.h"
/** PMCR16 nyaan 
 *  \par    About PMRC16 class
 *  - A Program with which to control Photo MOS Relay Circuit
 *  for 16 channels Electric Stimulation.
 *  - Photo Couplers are controlled by Shift Resisters. So this lib serves 
 *  shift resister's utility.
 *
 *  \par    Image of Photo MOS Relay Array
 *  -   32 Photo MOS Relay IN 
 *  -   16 channel OUT 
 *  
 *  \verbatim
 *   +--i      o----+----i      o--+
 *   |  [PMR01-1]   |    [PMR01-2] |
 *   |         (ch1(B)out)         |
 *   |                             |
 *   +--i      o----+----i      o--+
 *   |  [PMR02-1]   |    [PMR02-2] |
 *   |         (ch2(R)out)         |
 *   |                             |
 *   .              .              .
 *   .              .              .
 *   .              .              .
 *  Vpp                           GND
 *  \endverbatim
 *
 *  \par Info. Shit-Resister circuit board
 *  -   nOE is connected to GND
 *
 *  \verbatim
 *  in
 *  ->[PMR01-1]->[PMR01-2]->[PMR02-1]->[PMR02-2]
 *  ->[PMR03-1]->[PMR03-2]->[PMR04-1]->[PMR04-2]
 *  ->[PMR05-1]->[PMR05-2]->[PMR06-1]->[PMR06-2]
 *  ->[PMR07-1]->[PMR07-2]->[PMR08-1]->[PMR08-2]
 *  ->[PMR09-1]->[PMR09-2]->[PMR10-1]->[PMR10-2]
 *  ->[PMR11-1]->[PMR11-2]->[PMR12-1]->[PMR12-2]
 *  ->[PMR13-1]->[PMR13-2]->[PMR14-1]->[PMR14-2]
 *  ->[PMR15-1]->[PMR15-2]->[PMR16-1]->[PMR16-2]
 *  ->out
 *  \endverbatim
 *
 *  \version 1
 *      -   2016/dec/11-
 *      - program bersion for 8 channel circuit
 *  \version 16.1
 *      -   2018/jun/01-
 *      -   16 channel version
 *  \version 16.2
 *      -   2018/jun/28-
 *      -   major change
 *  \version 16.x
 */
class PMRC16ch
{
public:
    //
    //  Constructors
    //
    /// Constructor default
    PMRC16ch();
    /// Constructor with setting num of channels
    PMRC16ch(
        uint8_t arg_num_ch  ///<[in] number of channels to use
    );
    /// Construxtor with full configuration
    PMRC16ch(
        uint8_t arg_num_ch, ///<[in] number of channels to use
        PinName arg_SCK,    ///<[in] pin of shift registor clock
        PinName arg_CLR,    ///<[in] pin to clear the shift registor
        PinName arg_RCK,    ///<[in] pin of storage registor (buffer) clock
        PinName arg_SER,    ///<[in] pin to insert data
        PinName arg_OUT     ///<[in] pin to receive overflowed bit
    );

    //  Const.
    enum State {
        ALLGROUND = 0x55555555,
        CH1 = /*1V*/0b10000000000000000000000000000000/*16G*/,
        CH2 = /*1V*/0b00100000000000000000000000000000/*16G*/,
        CH3 = /*1V*/0b00001000000000000000000000000000/*16G*/,
        CH4 = /*1V*/0b00000010000000000000000000000000/*16G*/,
        CH5 = /*1V*/0b00000000100000000000000000000000/*16G*/,
        CH6 = /*1V*/0b00000000001000000000000000000000/*16G*/,
        CH7 = /*1V*/0b00000000000010000000000000000000/*16G*/,
        CH8 = /*1V*/0b00000000000000100000000000000000/*16G*/,
        CH9 = /*1V*/0b00000000000000001000000000000000/*16G*/,
        CH10= /*1V*/0b00000000000000000010000000000000/*16G*/,
        CH11= /*1V*/0b00000000000000000000100000000000/*16G*/,
        CH12= /*1V*/0b00000000000000000000001000000000/*16G*/,
        CH13= /*1V*/0b00000000000000000000000010000000/*16G*/,
        CH14= /*1V*/0b00000000000000000000000000100000/*16G*/,
        CH15= /*1V*/0b00000000000000000000000000001000/*16G*/,
        CH16= /*1V*/0b00000000000000000000000000000010/*16G*/,
        ALLHiZ    = 0x00000000
    };
    enum Polarity {Anodic = 0, Cathodic = 1};
    enum StimMode {ONE_VS_THEOTHERS, TWIN_ELECTRODES};
    //
    //  Function to prepare channels
    //
    void allGround();
    void allHiZ();
    void setPol(Polarity pol);  //inline definition
    void setOvsO(char ch);
    void setTwin(char stim_ch, char ref_ch);
    void setTrio(char stim_ch, char ref_ch1, char ref_ch2);
    //
    //  Function to get prameter
    //
    uint32_t getState();        //inline definition
    bool getPol();              //inline definition

private:
    //  The number of channels to use
    const uint8_t m_num_ch;
    //
    //  Sig var to controll PRM circuit
    //
    DigitalOut m_SCK;   //  Shift-resister's clock
    DigitalOut m_CLR;   //  To use when you want to clear shift-resister
    DigitalOut m_RCK;   //  FF's clock
    DigitalOut m_SER;   //  Serial input to shift data in a shift-resister
    DigitalIn  m_OUT;   //  Output data overflowed from shift-resister
    //
    // Var of state
    //
    //  Position of stimulation bits
    //  0: no stimbits, 1~m_num_ch: channel of stimulation
    uint32_t m_pos_stim;
    static const uint32_t m_statearray[];
    uint32_t m_PMRC_state;
    Polarity m_PMRC_POL;
    StimMode m_PMRC_mode;
    //  Initialization
    void init();
    //
    //  Function to control buffer of shift-resister
    //
    //  shift the bits arbitary times
    void shiftby(int times);
    //  the function with which to rapidly set it all-ground
    void sweep();
    //  set the simulation bits of 01
    void setStimbits();
    //  set all bits arbitarily
    void setBits(const uint32_t bits);
    //
    //  Clock Function
    //
    //update the clock of shift-resister
    void update();
    //  output the data by uploading them to FF Resitors
    void upload();
};
inline void PMRC16ch::setPol(Polarity arg_pol)
{
    m_PMRC_POL = arg_pol;
}
inline uint32_t PMRC16ch::getState()
{
    return m_PMRC_state;
}
inline bool PMRC16ch::getPol()
{
    //return false if cathodic
    //return true  if anodic
    return static_cast<bool>(m_PMRC_POL);
}
#endif