/** HV507 boards' utilities
 *
 *  This program is for using HV507(Shift Register Chip Circuit) for
 *  electric stimulation. The base Program had been written by Kajimoto Hiroyuki.
 *
 *  \file   KajiLabHV507B.h
 *  \Author Akifumi TAKAHASHI
 *
 *  \date   2015/Jul/16th Ver.1
 *  \date   2018/Fall Kajimoto-sensei was promoted to Professor!!!
 *  \date   2018/Nov/27th Ver.2 Simplified as a pure channel switching program.
 */
#ifndef MBED_HV507DRIVER_H
#define MBED_HV507DRIVER_H

#include "mbed.h"

/*  --------------------------------------------------------------------------------------------------------
 *  HV507 Function table
 *  --------------------------------------------------------------------------------------------------------
 *  Function        Input                                           Output
 *                  Data    CLK     !LE     !BL     !POL    DIR     Shift Reg       HV Out          Data Out
 *                                                                  1       2~64    1       2-64
 *  ========================================================================================================
 *  All on          X       X       X       L       L       X       *       *~*     H       H-H     *
 *
 *  All off         X       X       X       L       H       X       *       *~*     L       L-L     *
 *
 *  Invert mode     X       X       L       H       L       X       *       *~*     !*      !*~!*   *
 *  --------------------------------------------------------------------------------------------------------
 *  Load S/R        H or L  up      L       H       H       X       H or L  *~*     *       *~*     *
 *
 *  Store data      X       X       down    H       H       X       *       *~*     *       *~*     *
 *    in latches    X       X       down    H       L       X       *       *~*     !*      !*~!*   *   
 *
 *  Transparent     L       up      H       H       H       X       L       *~*     L       *~*     *
 *    latch mode    H       up      H       H       H       X       H       *~*     H       *~*     *
 *  --------------------------------------------------------------------------------------------------------
 *  I/O Relation    DIOA    up      X       X       X       L       Qn ->   Qn+1        -           DIOB
 *                  DIOB    up      X       X       X       H       Qn ->   Qn+1        -           DOIA
 */
 
 /**    HV507 class
  *
  *     With this class you can deal a HV507 board'''s''' as one object
  */ 
class HV507
{
public:
    /** Constructor
     *
     *  \param (const int arg_num_of_board = 1) the number of Hv507 board.
     */
    HV507(const int arg_num_board = 1);
    
    /** HV507 initialization
     *
     *  This is curried in constructor.
     *  In this function, all output pins are set as L (ALL GROUND),
     *  and Porarity (POL) is set to 1 (nomal mode; in other words, 
     *  data in latches of HV507 is reflected to output as is.
     *  If POL = 0, the output is inverted data (inverse mode).
     */
    void init();   
    
    /** Set a stim bit.
     *  
     *  Set one high bit (1) this is regarded as stim bit.
     */
    void setHtoSR();
    
    /** Set a ground bit
     *  
     *  Set one low bit (0) this is regarded as ground bit.
     *  If shifting the register many times, it is recommended to use 
     *  shiftSRBits_by().
     */
    void setLtoSR();
    
    /** Function to shift data only index-incresing-direction
     *
     *  \param (int arg_num_shifting) stimulation point
     */
    void shiftSRBits_by(int arg_num_shifting);
    
    /** Fill all with zero (ground)
     */
    void clearSR();
    
    /** Function to define ONE stimulation point
     *
     *  \param (const int arg_ch) the pin number supposed to be a stim pin.
     */
    void setCh(const int arg_ch);
    
    /** Function to store data in shift register to latches.
     *
     *  This is called in setCh. But if you need to use other function to deal
     *  the shift rester, you should call this function to output the data.
     */
    void updateLatches();
    
    void DEBUG_ALLPIN_ON();
    void DEBUG_ALLPIN_OFF();
    
private:
    //  HV507 Logical Pins
    //  Logical Pins Array = (DIOB, CLK, LE, BL, POL)
    DigitalOut  m_DIOB; ///< Pin to transfer Data input to HV507 
    DigitalOut  m_BL;   ///< Pin to transfer signal !BL(not-Blaker)
    DigitalOut  m_POL;  ///< Pin to transfer signal !POL(not-Pole signal)
    DigitalOut  m_CLK;  ///< Pin to transfer signal Clock
    DigitalOut  m_LE;   ///< Pin to transfer signal !LE(not-Latch Enable)
    
    /** The number of all pins including in Hv507 boards.
     * This is culculated in a constructor
     */
    const int   m_num_pins; 
    
    /** The constant number that expresses the number of output pins in a HV507
     */
    //enum :int   {HV507_NUM_PINS = 64};
    static const int HV507_NUM_PINS = 64;
    
    /** The variable expressing the postion of stimulation point
     *
     *  -    0   : No stimulationpoint
     *  -    1 ~ (HV507_PIN_NUM * m_num_of_board): a stimulation point;
     *       if the number of stim points are over than 1, this var means
     *       the last one.
     */
    int m_pos_stim;
    
    
    void putSRCLKahead();
};
#endif