/**
 *  \file   "KajiLabHV507B.cpp"
 *  \Author Akifumi TAKAHASHI
 */

#include "KajiLabHV507B.h"

HV507::HV507(const int arg_num_board):
    m_DIOB(p14),    //  Pin to transfer Data input to HV507
    m_BL(p13),      //  Pin to transfer signal -BL(Blaker)
    m_POL(p12),     //  Pin to transfer signal POL
    m_CLK(p11),     //  Pin to transfer signal Clock
    m_LE(p10),      //  Pin to transfer signal -LE(Latch Enable)
    m_num_pins(arg_num_board * HV507_NUM_PINS)
{
    init();
}

void HV507::DEBUG_ALLPIN_ON()
{
    m_DIOB  = 1;
    m_BL    = 1;
    m_POL   = 1;
    m_CLK   = 1;
    m_LE    = 1;
}
void HV507::DEBUG_ALLPIN_OFF()
{
    m_DIOB  = 0;
    m_BL    = 0;
    m_POL   = 0;
    m_CLK   = 0;
    m_LE    = 0;
}

//HV507 init
void HV507::init()   // argument: the number of Hv507. #aktk150715
{

    //  First, Rapidly Temporarily Make All Outputs Off (Low)
    m_POL = 1;  //  data in latches is reflected to the output as is (normal mode).
    m_BL  = 0;  //  L == ALL OFF (ShiftReg not cahnges, HVouts be all Low, DATAout not changes)
    m_LE  = 0;  //  Close the latches

    m_DIOB = 0; //  input data = L
    for(int pin = 0; pin < m_num_pins; pin++) putSRCLKahead();
    m_pos_stim = 0;
    
    updateLatches();

    m_BL = 1;
}

void HV507::shiftSRBits_by(int arg_num_shifting)
{
    //  Load S/R Mode ON
    m_BL = 1;
    m_LE = 0;

    //  Fill 0 into ShitRegisters before supposing node to Shift 1 to the node.
    m_DIOB = 0;
    for(int itr = 0; itr < arg_num_shifting; itr++) putSRCLKahead();
    
    m_pos_stim += arg_num_shifting; 
    if(m_pos_stim > m_num_pins) m_pos_stim = 0;
}

void HV507::setHtoSR()
{
    m_BL = 1;
    m_LE = 0;
    
    m_DIOB = 1;
    putSRCLKahead();
    m_pos_stim = 1;
}

void HV507::setLtoSR()
{
    m_BL = 1;
    m_LE = 0;
    
    m_DIOB = 0;
    putSRCLKahead();
    m_pos_stim >= m_num_pins? m_pos_stim = 0 : m_pos_stim++;
    
}

void HV507::putSRCLKahead()
{
    m_CLK = 0;  //  Clock down(L)
    m_CLK = 0;  //
    //  Load a datum to the SR(s) when clock rising
    m_CLK = 1;  //  Clock up(H)
    m_CLK = 1;  //  (twice substitution is for mbed faster than HV507.
}



//  Fill out 0 into all ShiftRegs.
void HV507::clearSR()
{
    if(m_pos_stim != 0) shiftSRBits_by(m_num_pins + 1 - m_pos_stim);
}

void HV507::setCh(const int arg_ch)
{
    if(arg_ch <= 0 || m_num_pins < arg_ch)
        clearSR();
    else {
        if(arg_ch < m_pos_stim){
            clearSR();
            setHtoSR();
            shiftSRBits_by(arg_ch - 1);
        }
        else
            shiftSRBits_by(arg_ch - m_pos_stim);
    }
    updateLatches();
}

void HV507::updateLatches()
{
    //  Make data to be stored in Latches
    //  Reflect the data in SR(s) to Latches
    m_LE = 1;   
    m_LE = 1;
    //  Close latch and store the data in it
    m_LE = 0;   
    m_LE = 0;
}


/*--------------------------------------------------------
    Example Main Functin For Simple Comunication
----------------------------------------------------------*/
/*
DigitalOut dout(p14); //trigger signal for check
Serial pc(USBTX, USBRX); // tx, rx
AnalogIn volume(p15); //volume
#define HV507_NUM 1
#define ELECTRODE_NUM 61
#define PC_MBED_STIM_PATTERN 0xFF
#define PC_MBED_MEASURE_REQUEST 0xFE
#define MBED_PC_MEASURE_RESULT 0xFF

int main()
{
    int pin;
    short PulseHeight, AD;
    unsigned char stim_pattern[ELECTRODE_NUM]={0},uc;
    unsigned char impedance[ELECTRODE_NUM]={0};

    pc.baud(921600);
    DAADinit();
    hv507Init(1); //initialize one HV507

    while (1) {
        //pc.printf("%d\n", (int)PulseHeight);
        if (pc.readable()) { //check serial buffer
            uc=pc.getc();
            if (uc == PC_MBED_STIM_PATTERN) {//if PC requires stimulation
                for (pin=0; pin<ELECTRODE_NUM; pin++) {//read data
                    stim_pattern[pin] =pc.getc();
                    if(stim_pattern[pin]>100)stim_pattern[pin]=100; //limit
                }
                PulseHeight = (short) (volume*1024.0); //0-5mA
                hv507FastScan(0);
                for (pin=0; pin<ELECTRODE_NUM; pin++) {
                    if(stim_pattern[pin]!=0) {
                        if(pin!=0) {
                            hv507FastScan(pin);
                        }
                        m_BL = 1;
                        m_LE = 1;
                        m_LE = 0;

                        //Cathodic Stimulation
                        //m_POL = 0;

                        //50us stimulation
                        AD = DAAD(PulseHeight); //0-5mA. Simultaneous DA and AD. 2.0us
                        wait_us(stim_pattern[pin]); //0-100us
                        AD = DAAD(0);
                        m_BL = 0;
                        wait_us(100-stim_pattern[pin]); //wait
                    }
                }
                hv507Clear(HV507_NUM);    //cleaning
                m_LE = 1;
                m_LE = 0;
                m_BL = 0;
            }else if (uc == PC_MBED_MEASURE_REQUEST) {//if PC requires stimulation
                PulseHeight = 100.0; //0.5mA
                hv507FastScan(0);
                //50us stimulation
                m_BL = 1;
                m_LE = 1;
                m_LE = 0;
                AD = DAAD(PulseHeight); //0-5mA. Simultaneous DA and AD. 2.0us
                wait_us(30);
                AD = DAAD(0);
                m_BL = 0;
                wait_us(70);
                for (pin=0; pin<ELECTRODE_NUM; pin++) {
                        if(pin!=0) {
                            hv507FastScan(pin);
                        }
                        m_BL = 1;
                        m_LE = 1;
                        m_LE = 0;

                        //50us stimulation
                        AD = DAAD(PulseHeight); //0-5mA. Simultaneous DA and AD. 2.0us
                        wait_us(30);
                        AD = DAAD(0);
                        impedance[pin]=AD>>3;
                        if(impedance[pin]>200)impedance[pin]=200;
                        m_BL = 0;
                        wait_us(70);
                }
                hv507Clear(HV507_NUM);    //cleaning
                m_LE = 1;
                m_LE = 0;
                m_BL = 0;
                pc.putc(MBED_PC_MEASURE_RESULT);
                for (pin=0; pin<ELECTRODE_NUM; pin++) {
                    pc.putc(impedance[pin]);
                }
            }
        }
    }
}
*/
