/** Defining Amplide Modulated Pulse Train Model
 *
 *  \file   AMPulseTrain.h
 *  \author Akifumi Takahashi
 *  \date   2019/12/02 -
 *  \version 1.0.2019.Dec
 */

#ifndef AM_PULSE_TRAIN_H
#define AM_PULSE_TRAIN_H


#include "mbed.h"
#include "AMSignal.h"
#include "PulseTrain.h"

/** \Class Amplide Modulated Pulse Train Model
 *
 *  Pulse Train Model which clock is defined in scale of us;
 *  the model pulses' height can be modulated as a product with m_ampl and any function
 *  returning a int as the coefficeincy.
 *  You can define the carrier pulse train's freq and duty cycle like PWM.
 *
 */
class AMPulseTrain
{
public:
    /** Constractor
     */
    AMPulseTrain(
        /// Carrier Pulse Train
        PulseTrain  * arg_carrier = new PulseTrain(),
        /// Initial AM Signal expression
        AMSignal  * arg_signal = new AMSignal()
    );

    /// Carrier Pulse Train
    PulseTrain  * Carrier;

    /// AM Signal
    AMSignal  * Signal;

    /// register callback called every clock (not carrier pulse edges)
    void attachCallback_asClock(
        Callback<void(bool,  AMPulseTrain*)> arg_callback
    );

    /// register callback called every carrier pulse edges
    void attachCallback_asPulseEdge(
        Callback<void(bool,  AMPulseTrain*)> arg_callback
    );

    /// register callback as AMSignalExpression
    void attachAMSignalExpression(
        Callback<uint16_t(AMPulseTrain*)> arg_callback
    );

    void setFrequency_Carrier( uint32_t const arg_freq )
    {
        Carrier->setFrequency(arg_freq);
        RecalcPulsebaseParameter();
    }

    void setFrequency_Signal( uint32_t const arg_freq )
    {
        Signal->setFrequency(arg_freq);
        RecalcPulsebaseParameter();
    }

    uint32_t getFrequency_Carrier()
    {
        return Carrier->getFrequency();
    }

    uint16_t getClockperiod_us();

    void incrementClock();

    uint16_t getPeriod_pPulse_Signal()
    {
        return m_AMSIGNAL_PERIOD_PER_PULSE;
    }

    uint16_t getPWidth_pPulse_Signal()
    {
        return m_AMSIGNAL_PWIDTH_PER_PULSE;
    }

private:

    Callback<void(bool,  AMPulseTrain*)> m_callback_asClock;
    void CallbackWrapper_asClock(bool arg_pulsestate);

    Callback<void(bool,  AMPulseTrain*)> m_callback_asPulseEdge;
    void CallbackWrapper_asPulseEdge(bool arg_pulsestate);

    Callback<uint16_t(AMPulseTrain*)>  m_AMSignalExpression;
    uint16_t CallbackWrapper_AMSignalExpression(AMSignal* arg_signal);

    uint16_t m_AMSIGNAL_PERIOD_PER_PULSE;
    uint16_t m_AMSIGNAL_PWIDTH_PER_PULSE;
    void RecalcPulsebaseParameter()
    {
        m_AMSIGNAL_PERIOD_PER_PULSE = Signal->getPeriod_us() / Carrier->getPeriod_us();
        m_AMSIGNAL_PWIDTH_PER_PULSE = 500 / Carrier->getPeriod_us();
    }
};

#endif