/** Defining Amplide Modulated Pulse Train Model
 *
 *  \file   PulseTrain.h
 *  \author Akifumi Takahashi
 *  \date   2019/12/02 -
 *  \version 1.0.2019.Dec
 */

#ifndef PULSE_TRAIN_H
#define PULSE_TRAIN_H

#include "mbed.h"
/** \Class Pulse Train Model
 *
 *  Pulse Train Model which clock is defined in scale of [us];
 *  You can define the carrier pulse train's freq and duty cycle like PWM.
 *  Kinds of Frequency is dealed as unsigned 32bit int
 *  For the other int vars, allocated is 32bit.
 */
class PulseTrain
{
public:
    /** Constractor
     */
    PulseTrain(
        /// Initial carrier pulse frequency
        uint32_t const arg_freq_init = 4000,
        /// Initial carrier pulse duty cycle
        float const arg_duty_init = 0.5,
        /// Initialize FREQ_MAX
        uint32_t const arg_freq_max = 8000
    );
    /** Increment the clock to let go ahead the wave state
     *
     *  If callback as rising/falling is also called,
     *  this "asClock" is called earlier
     */
    void incrementClock();

    /** Executes a callback fanction called as clock is incremented
     *
     *  If callback as rising/falling is also called,
     *  this "asClock" is called at the last.
     */
    void attachCallback_asClock(
        /** Called back as clock incremented
         *
         *  \arg <pulsestate> indicate whther pulse
         *  (not clock pulse but one of pulse train) has risen/fallen
         */
        Callback<void(bool)> arg_callback
    );

    void attachCallback_asPulseEdge(
        /** Called back as a pulse rising/falling
         *
         *  \arg <pulsestate> indicate whther pulse has risen/fallen
         */
        Callback<void(bool)> arg_callback
    );

    void setFrequency(uint32_t const arg_freq);

    void setDutycycle(float const arg_duty);

    bool
    getState();         //inline
    uint32_t
    getFrequency();     //inline
    float
    getDutycycle();     //inline
    uint32_t
    getPeriod_us();   //inline
    uint32_t
    getClockperiod_us();   //inline


    uint32_t const FREQ_MAX;

    template <typename T>
    static T velidateRange(T const arg_val, T const arg_min, T const arg_max);

private:
    void init();
    
    /// Frequency of the Carrier Pulse (Hz)
    uint32_t m_freq;

    /// Duty cycle
    float m_duty;
    
    /// Period
    uint32_t m_period_us;

    /// GCD of the pulse period and pulse width
    uint32_t m_clock_period_us;
    
    /** Period per clock period
     *
     *  The unit is what times the clock tickes
     */
    uint32_t m_period_pcp;
    
    /// Timing [us] a pulse rising within a period [us]
    static uint32_t const m_raising = 0;
    
    /** Timing [us] a pulse falling within a period [us]
     *
     *  Calculated in init()
     */
    uint32_t m_falling;

    /// Flag if a palse is raised (High) or not (Low)
    bool m_pulsestate;

    /** Called back as clock incremented
     *
     *  \arg <pulsestate> indicate whther pulse
     *  (not clock pulse but one of pulse train) has risen/fallen
     */
    Callback<void(bool)> m_callback_asClock;

    /** Called back as a pulse rising/falling
     *
     *  \arg <pulsestate> indicate whther pulse has risen/fallen
     */
    Callback<void(bool)> m_callback_asPulseEdge;

    static void doNothing(bool arg_b) {;}

};
#endif