/** Defining Amplide Modulated Pulse Train Model
 *
 *  \file   AMSignal.h
 *  \author Akifumi Takahashi
 *  \date   2019/12/02 -
 *  \version 1.0.2019.Dec
 */

#ifndef AM_SIGNAL_H
#define AM_SIGNAL_H


#include "mbed.h"

/** \Class Amplide Modulated Signal Model for AM Pulse Train Model
 *
 *  All parameter data is dealed as unsigned 16 bit int.
 *  \version alluint16
 */
class AMSignal
{
public:
    /// Constructor
    AMSignal(
        uint16_t const arg_freq = 50,
        uint16_t const arg_freq_max = 100,
        uint16_t const arg_ampl_max_u16 = 4095
    ):
        FREQ_MAX(arg_freq_max),
        AMPL_MAX_u16(arg_ampl_max_u16)
    {
        setFrequency(arg_freq);
        attachAMSignalExpression(constantDC_defaltSignal);
    }

    /** Amplitude Modified Signal / Pulse Hight Writing Handler
     *
     *  For fast processing, no float culc nor div culc is recommended
     */
    void attachAMSignalExpression(
        /** Callback
         *
         *  \arg AMSignal* contains parameters of this.
         *  \retval is dealed as a factor the argument of Callback as Pulse rising.
         */
        Callback<uint16_t(AMSignal*)> arg_signal
    )
    {
        m_AMSignalExpression = arg_signal;
    }

    uint16_t getAMSinalValue()
    {
        return m_AMSignalExpression(this);
    }


private:
    uint16_t m_ampl_u16;
    uint16_t m_freq;
    uint16_t m_period_us;
    template <typename T>
    static T velidateRange(T const arg_val, T const arg_min, T const arg_max);

    Callback<uint16_t(AMSignal*)> m_AMSignalExpression;

    /// Default amplitude modulation signal
    static uint16_t constantDC_defaltSignal (AMSignal* arg)
    {
        return 1;
    }


public:
    //  constants
    uint16_t const FREQ_MAX;
    uint16_t const AMPL_MAX_u16;

    /** Frequency could be slightly modulated because it depends on the carrier freq
     */
    void setFrequency(uint16_t const arg_freq);

    /** Set Amplitude Paramiter
     *
     *  -   Crumping a value within [0f,1f], converted to [0, 0xFFFF]
     *  -   Note that this is a parameter which affects a pulse hight,
     *  but isn't the pulse hight itself
     *  -   The Pulse hight is defined in callback attached with attachAMSignalExpression()
     */
    void setAmplitude(float const arg_ampl);

    void setAmplitude(uint16_t const arg_ampl);

    /// Get a parameter which defines the size of pulse hight axis with in [0,1]
    float getAmplitude_uf();       //inline

    /// Get a parameter which defines the size of pulse hight axis with in [0, 4096]
    uint16_t getAmplitude_u16(); //inline

    uint16_t getFrequency();    //inline

    uint16_t getPeriod_us();    //inline

};
#endif