/*
 *  2014/12/16 Modified by akkera102
 *
 *  DHT Library for  Digital-output Humidity and Temperature sensors
 *
 *  Works with DHT11
 *             SEN11301P,  Grove - Temperature&Humidity Sensor     (Seeed Studio)
 *
 *  Copyright (C) Wim De Roeve
 *                based on DHT22 sensor library by HO WING KIT
 *                Arduino DHT11 library
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documnetation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to  whom the Software is
 * furished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS OR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mbed.h"

Serial pc(USBTX, USBRX);
DigitalInOut pin(p8);

enum {
    SUCCESS,
    ERROR_TIMEOUT,
    ERROR_BUS_BUSY,
    ERROR_NOT_PRESENT,
    ERROR_NO_PATIENCE,
    ERROR_SYNC_TIMEOUT,
    ERROR_DATA_TIMEOUT,
    ERROR_CHECKSUM
};

float Temperature;
float Humidity;


int Stall(int usTimeMax, int level)
{
    int usTime = 0;

    while(usTime < usTimeMax)
    {
        if(pin == level)
        {
            return SUCCESS;
        }

        usTime++;
        wait_us(1);
    }

    return ERROR_TIMEOUT;
}

int WaitPinHigh(int usTimeOut)
{
    return Stall(usTimeOut, 1);
}

int WaitPinLow(int usTimeOut)
{
    return Stall(usTimeOut, 0);
}

int ReadDHT11(void)
{
    // IO must be in hi state to start
    if(WaitPinHigh(250) == ERROR_TIMEOUT)
    {
        return ERROR_BUS_BUSY;
    }

    // start the transfer
    pin.output();
    pin = 0;
    wait_ms(18);

    pin = 1;
    wait_us(30);

    pin.input();

    // wait till the sensor grabs the bus
    if(WaitPinLow(40) == ERROR_TIMEOUT)
    {
        return ERROR_NOT_PRESENT;
    }

    // sensor should signal low 80us and then hi 80us
    if(WaitPinHigh(100) == ERROR_TIMEOUT)
    {
        return ERROR_SYNC_TIMEOUT;
    }
    if(WaitPinLow(100) == ERROR_TIMEOUT)
    {
        return ERROR_NO_PATIENCE;
    }

    int i, bit;
    uint8_t buf[5];

    // capture the data(40 bit)
    for(i=0; i<5; i++)
    {
        buf[i] = 0;

        for(bit=0; bit<8; bit++)
        {
            if(WaitPinHigh(75) == ERROR_TIMEOUT)
            {
                return ERROR_DATA_TIMEOUT;
            }

            // logic 0 is 28us max, 1 is 70us
            wait_us(40);
            buf[i] |= pin << (7 - bit);

            if(WaitPinLow(50) == ERROR_TIMEOUT)
            {
                return ERROR_DATA_TIMEOUT;
            }
        }
    }

    for(i=0; i<5; i++)
    {
        printf("buf[%d] = 0x%02x\n", i, buf[i]);
    }

    // valid check
    if(buf[4] != buf[0] + buf[1] + buf[2] + buf[3])
    {
        return ERROR_CHECKSUM;
    }

    Temperature = float(buf[2]);
    Humidity    = float(buf[0]);

    return SUCCESS;
}

int main(void)
{
    int ret;

    for(;;)
    {
        ret = ReadDHT11();

        if(ret == SUCCESS)
        {
            printf("Temperature : %4.2f\n", Temperature);
            printf("Humidity    : %4.2f\n", Humidity);
        }
        else
        {
            printf("Error : %d\n", ret);
        }

        wait(2);
    }
}
