/** DRV2665 library
 *
 * @author Akash Vibhute
 * @author < akash . roboticist [at] gmail . com >
 * @version 0.1
 * @date May/24/2016
 *
 * @section LICENSE
 *
 * Copyright (c) 2015 Akash Vibhute
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 */

/* Contol registers */
#define DRV2665_STATUS  0x00
#define DRV2665_CTRL_1  0x01
#define DRV2665_CTRL_2  0x02
#define DRV2665_FIFO    0x0b

/* Status Register */
#define DRV2665_FIFO_FULL       0x00
#define DRV2665_FIFO_EMPTY      0x02

/* Control 1 Register */
#define DRV2665_25_VPP_GAIN     0x00
#define DRV2665_50_VPP_GAIN     0x01
#define DRV2665_75_VPP_GAIN     0x02
#define DRV2665_100_VPP_GAIN    0x03
#define DRV2665_DIGITAL_IN      0xfb //0x00
#define DRV2665_ANALOG_IN       0x04 

#define DRV2665_GAIN_RD_MASK    0x03
#define DRV2665_IN_RD_MASK      0x04 
#define DRV2665_CHIPID_RD_MASK  0x78


/* Control 2 Register */
#define DRV2665_BOOST_EN            0x02
#define DRV2665_STANDBY             0x40
#define DRV2665_DEV_RST             0x80
#define DRV2665_5_MS_IDLE_TOUT      0x00
#define DRV2665_10_MS_IDLE_TOUT     0x04
#define DRV2665_15_MS_IDLE_TOUT     0x08
#define DRV2665_20_MS_IDLE_TOUT     0x0c

#define DRV2665_I2C_ADDRESS         0x59

#ifndef DRV2665_H
#define DRV2665_H
 
#include "mbed.h"
 
/** DRV2665_DIGITAL class
 *
 *  @section DESCRIPTION
 *  DRV2665: Piezo Haptic Driver with Integrated Boost Converter and Digital Front End
 *  The DRV2665 device is a piezo haptic driver with
 *  integrated 105-V boost switch, integrated power
 *  diode, integrated fully-differential amplifier, and
 *  integrated digital front end. This versatile device is
 *  capable of driving both high-voltage and low-voltage
 *  piezo haptic actuators. The input signal can be driven
 *  as haptic packets over the I2C port or through the
 *  analog inputs
 *
 *  The digital interface of the DRV2665 device is
 *  available through an I2C compatible bus. A digital
 *  interface relieves the costly processor burden of the
 *  PWM generation or additional analog channel
 *  requirements in the host system. Any writes to the
 *  internal first-in, first-out buffer (FIFO) will
 *  automatically wake up the device and begin playing
 *  the waveform after the 2 ms internal start-up
 *  procedure. When the data flow stops or the FIFO
 *  under-runs, the DRV2665 device will automatically
 *  enter a pop-less shutdown procedure.
 *
 *  The boost voltage is set using two external resistors,
 *  and the boost current limit is programmable through
 *  the REXT resistor. A typical start-up time of 2 ms
 *  makes the DRV2665 device an ideal piezo driver for
 *  fast haptic responses. Thermal overload protection
 *  prevents the device from being damaged when
 *  overdriven.
 *
 *  Datasheet for DRV2665 can be found at:
 *  http://www.ti.com/lit/gpn/drv2665
 *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "DRV2665.h"
 *  
 *  DRV2665_DIGITAL piezo(I2C_SDA, I2C_SCL); // DRV2665 digital mode object
 *  
 *  int main()
 *  {
 *      piezo.init(DRV2665_100_VPP_GAIN, DRV2665_5_MS_IDLE_TOUT); // set output level to 100Vpp and idle timeout to 5ms
 *
 *      while(1) {
 *          
 *          piezo.outputSine(8000); // outputs 8khz sine waveform at selected voltage level
 *
 *          wait_ms(1);
 *      }
 *  }
 *  @endcode
 */
 
class DRV2665_DIGITAL
{
public:
 
    /** Create a DRV2665_DIGITAL instance connected to specified I2C pins with specified address
     *
     * @param sda I2C-bus SDA pin
     * @param scl I2C-bus SCL pin
     */
    DRV2665_DIGITAL( PinName sda, PinName scl );
 
    /** Initializing DRV2665 in digital mode
     *
     * Switch to commended downstream I2C channel
     *
     * @param output_gain sets the output voltage level
     * @param idle_timeout sets the idle timeout in ms
     */
    void init(uint8_t output_gain, uint8_t idle_timeout);
 
    /** Reset DRV2665
     *
     *  Reset piezo driver
     */
    void reset();
    
    /** Output waveform
     *
     *  Writes custom output waveform to FIFO (100 bytes max), which is played back at 8kHz
     *  @param waveform[] contains the waveform values
     *  @param length tells the function how many values makeup the wave
     */
    void outputWave(int8_t waveform[], uint8_t length);
    
    
    /** Output sine
     *
     *  Writes sinewave values to FIFO buffer
     *  @param hz contains the frequency in hz of the waveform
     */
    void outputSine(uint16_t hz);
    
    /** Check FIFO status
     *
     *  returns the status of the FIFO buffer, false -> FIFO is full, true otherwise
     */
    bool fifo_check();
    
    /** Override boot converter
     *
     *  To override boot converter to be enabled indefinitely (-> true) or controlled by device logic (-> false)
     */
    void en_override(uint8_t en);
    
private:
    I2C i2c_;
    uint8_t i2c_addr;
    
    //Internal functions
    uint8_t read(char reg);
    void write(char reg, uint8_t data);
};
 
#endif  //  DRV2665