// Copyright 2020 Georgia Tech.  All rights reserved.
// The materials provided by the instructor in this course are for
// the use of the students currently enrolled in the course.
// Copyrighted course materials may not be further disseminated.
// This file must not be made publicly available anywhere.

#include "map.h"

#include "globals.h"
#include "graphics.h"

/**
 * The Map structure. This holds a HashTable for all the MapItems, along with
 * values for the width and height of the Map.
 */
struct Map {
    HashTable* items;
    int w, h;
};

#define NUM_MAPS 1
#define MAPS_WIDTH 50
#define MAPS_HEIGHT 50
static Map maps[NUM_MAPS];
static int active_map;
unsigned int NUMBUCKETS = 750;

// this is important
static const MapItem CLEAR_SENTINEL = {
    .type = CLEAR,
    .draw = draw_nothing
};

/**
 * The first step in HashTable access for the map is turning the two-dimensional
 * key information (x, y) into a one-dimensional unsigned integer.
 * This function should uniquely map (x,y) onto the space of unsigned integers.
 */
static unsigned XY_KEY(int X, int Y) {
     // multiply the map height by Y and adding it to X
     unsigned XYKey = Y * map_width() + X;
     // return the unique key
     return XYKey;
}

/**
 * This is the hash function actually passed into createHashTable. It takes an
 * unsigned key (the output of XY_KEY) and turns it into a hash value (some
 * small non-negative integer).
 */
unsigned map_hash(unsigned key)
{
    // modulo map hashing into something unique 
    return key % NUMBUCKETS;
}

/**
 * Initializes the internal structures for all maps. This does not populate
 * the map with items, but allocates space for them, initializes the hash tables, 
 * and sets the width and height.
 */
 
void maps_init()
{
    // TODO: Implement!    
    // Initialize hash table
    // Set width & height
    set_active_map(0)->items = createHashTable(map_hash, NUMBUCKETS); // num buckets
    get_active_map()->w = MAPS_WIDTH; 
    get_active_map()->h = MAPS_HEIGHT;
    return;
}

Map* get_active_map()
{
    return &maps[active_map];
}

Map* set_active_map(int m)
{
    active_map = m;
    return &maps[active_map];
}

void print_map()
{
    char lookup[] = {'W', 'D', 'P', 'A', 'K', 'C', 'N',' ','S'};
    Map* map = get_active_map();
    for(int j = 0; j < map->h; j++)
    {
        for (int i = 0; i < map->w; i++)
        {
            MapItem* item = (MapItem*)getItem(map->items, XY_KEY(i, j));
            if (item) pc.printf("%c", lookup[item->type]);
            else pc.printf(" ");
        }
        pc.printf("\r\n");
    }
}

/**
 * Returns the width of the active map.
 */
int map_width()
{
// get the pointer of the active map 
// and the pointer to the variable w
Map* map = get_active_map();
return map->w;
}

/**
 * Returns the heigh of the active map.
 */
int map_height()
{
// get the pointer of the active map
// and the pointer to the variable h
Map* map = get_active_map();
return map->w;
}

/**
 * Returns the total number of cells in the active map.
 */
int map_area()
{
// returns the product of height and width
return map_height() * map_width();
}

MapItem* get_current(int x, int y)
{
    // return the current item in that specific map
    return (MapItem*) get_active_map()->items;
}

/**
 * Returns the MapItem immediately above the given location.
 */
MapItem* get_north(int x, int y)
{
 // coordinate to the one block north of the current block
 int northCoordinate = y - 1;
 // get the items in the active map
 HashTable* mapItem = get_active_map()->items;
 // get the keys to the respective coordinate, placed in a specific bucket
 unsigned keys = XY_KEY(x, northCoordinate);  
 // we need to cast it because it is not type MapItem, but type HashTable
 return (MapItem*) getItem(mapItem, keys); 
}

/**
 * Returns the MapItem immediately below the given location.
 */
MapItem* get_south(int x, int y)
{
 // coordinate to the one block south of the current block
 int southCoordinate = y + 1;
 // get the items in the active map
 HashTable* mapItem = get_active_map()->items;
 // get the keys to the respective coordinate, placed in a specific bucket
 unsigned keys = XY_KEY(x, southCoordinate);  
 // we need to cast it because it is not type MapItem, but type HashTable
 return (MapItem*) getItem(mapItem, keys); 
}


/**
 * Returns the MapItem immediately to the right of the given location.
 */
MapItem* get_east(int x, int y)
{
 // coordinate to the one block east of the current block
 int eastCoordinate = x + 1;
 // get the items in the active map
 HashTable* mapItem = get_active_map()->items;
 // get the keys to the respective coordinate, placed in a specific bucket
 unsigned keys = XY_KEY(eastCoordinate, y);  
 // we need to cast it because it is not type MapItem, but type HashTable
 return (MapItem*) getItem(mapItem, keys); 
}


/**
 * Returns the MapItem immediately to the left of  the given location.
 */
MapItem* get_west(int x, int y)
{
 // coordinate to the one block west of the current block
 int westCoordinate = x - 1;
 // get the items in the active map
 HashTable* mapItem = get_active_map()->items;
 // get the keys to the respective coordinate, placed in a specific bucket
 unsigned keys = XY_KEY(westCoordinate, y);  
 // we need to cast it because it is not type MapItem, but type HashTable
 return (MapItem*) getItem(mapItem, keys); 
}

/**
 * Returns the MapItem at the given location.
 */
MapItem* get_here(int x, int y)
{
 // get the items in the active map
 HashTable* mapItem = get_active_map()->items;
 // get the keys to the respective coordinate, placed in a specific bucket
 unsigned keys = XY_KEY(x, y);  
 // we need to cast it because it is not type MapItem, but type HashTable
 return (MapItem*) getItem(mapItem, keys); 
}


/**
 * If there is a MapItem at (x,y), remove it from the map.
 */
void map_erase(int x, int y)
{
    if (get_here(x, y) != NULL) {
    return deleteItem(get_active_map()->items, XY_KEY(x, y));
        }
    return;
}

void add_wall(int x, int y, int dir, int len)
{
    for(int i = 0; i < len; i++)
    {
        MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
        w1->type = WALL;
        w1->draw = draw_wall;
        w1->walkable = false;
        w1->data = NULL;
        unsigned key = (dir == HORIZONTAL) ? XY_KEY(x+i, y) : XY_KEY(x, y+i);
        void* val = insertItem(get_active_map()->items, key, w1);
        if (val) free(val); // If something is already there, free it
    }
}

void add_plant(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = PLANT;
    w1->draw = draw_plant;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_goodie(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = GOODIE;
    w1->draw = draw_goodie;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void remove_goodie(int x, int y) // I'm lazy so overwrite it with a plant
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = PLANT;
    w1->draw = draw_plant;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_snake_body(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = SNAKE_BODY;
    w1->draw = draw_snake_body;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_snake_head(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = SNAKE_HEAD;
    w1->draw = draw_snake_head;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_snake_tail(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = SNAKE_TAIL;
    w1->draw = draw_snake_tail;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_shield(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = SHIELD;
    w1->draw = draw_shield;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_life(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = LIFE;
    w1->draw = draw_life;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_snowflake(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = SNOWFLAKE;
    w1->draw = draw_snowflake;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_sword(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = SWORD;
    w1->draw = draw_sword;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_nothing(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = NOTHING;
    w1->draw = draw_nothing;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_arrow_up(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = ARROWUP;
    w1->draw = draw_arrow_up;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_arrow_down(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = ARROWDOWN;
    w1->draw = draw_arrow_down;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_arrow_right(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = ARROWRIGHT;
    w1->draw = draw_arrow_right;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_arrow_left(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = ARROWLEFT;
    w1->draw = draw_arrow_left;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_lightspeed(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = LIGHTSPEED;
    w1->draw = draw_lightspeed;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_blue_wall(int x, int y, int dir, int len)
{
    for(int i = 0; i < len; i++)
    {
        MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
        w1->type = BLUEWALL;
        w1->draw = draw_blue_wall;
        w1->walkable = false;
        w1->data = NULL;
        unsigned key = (dir == HORIZONTAL) ? XY_KEY(x+i, y) : XY_KEY(x, y+i);
        void* val = insertItem(get_active_map()->items, key, w1);
        if (val) free(val); // If something is already there, free it
    }
}

void add_reverse(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = REVERSE;
    w1->draw = draw_reverse;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_blue_potion(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = BLUEPOTION;
    w1->draw = draw_blue_potion;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_red_potion(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = REDPOTION;
    w1->draw = draw_red_potion;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}

void add_portal(int x, int y) 
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = PORTAL;
    w1->draw = draw_portal;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y), w1);
    if (val) free(val); // If something is already there, free it
}
