

#ifndef _TSL45315_HPP_
#define _TSL45315_HPP_

#include "mbed.h"

#define I2C_ADDR        (0x52)
// Register definition
#define CONTROL_REG     0x00
#define CONFIG_REG      0x01
#define DATALOW_REG     0x04
#define DATAHIGH_REG    0x05
#define ID_REG          0x0A
//Control Register
#define POWER_DOWN      0X00
#define SINGLE_ADC      0x02
#define NORMAL_OP       0x03
//Config Register
#define M1              0x00
#define M2              0x01
#define M4              0x02

#define I2C_FREQ        400000

namespace TSL45x
{
/** Class: TSL45315
 *  Digital Ambient Light Sensor (ALS) - TSL45315
 *  Illumination values are directly presented in lux
 *  without calculation.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "tsl45315.hpp"
 
 * Serial pc(USBTX, USBRX);
 * //TSL45x::TSL45315 sensor(p9, p10);
 * //TSL45x::TSL45315 sensor(p9, p10, M1);
 * TSL45x::TSL45315 sensor(p9, p10, I2C_FREQ, 0.5);
 * //TSL45x::TSL45315 sensor(p9, p10, I2C_FREQ, 0.5, M1);
 *
 * int main( void )
 * {
 *   pc.printf("TSL45315 Illumination Sensor\n");
 *
 *   pc.printf("ID:\t%x\n", sensor.getID());
 *   pc.printf("Multiplier:\t%x\n", sensor.getMultiplier());
 *
 *   while(1) {
 *        //sensor.getLuxData();
 *        pc.printf("Illumination: %u lux\n",sensor.getLux());
 *
 *        wait(1);
 *    }
 * }
 * @endcode
 */
class TSL45315
{

private:

    I2C _i2c;
    Ticker _luxTicker;

    uint32_t lux;
    uint8_t devID;
    uint8_t multiplier;
    
public:
    /** Create a TSL45315 instance without variable integration time ( set to M1)
    *   with integrated ticker to repeatedly update the illumination value. 
    * @param sda - I2C Dataline pin
    * @param scl - I2C Clockline pin
    * @param i2c_freq - choose the i2c frequency
    * @param t - ticker interval in seconds
    */
    TSL45315(PinName sda, PinName scl, int i2c_freq, float t);
    
    /** Create a TSL45315 instance with variable integration time (M1,M2,M4)  
    *   and with integrated ticker to repeatedly update the illumination value. 
    * @param sda - I2C Dataline pin
    * @param scl - I2C Clockline pin
    * @param i2c_freq - choose the i2c frequency
    * @param mult - Multiplier with corresponding integration time (M1 = 400ms, M2 = 200ms, M4 = 100ms)
    * @param t - ticker interval in seconds
    */
    TSL45315(PinName sda, PinName scl, int i2c_freq, float t, uint8_t mult);
    
    /** Create a TSL45315 instance with variable integration time (M1,M2,M4)  
    *   without integrated ticker to repeatedly update the illumination value. 
    *   For use with seperate ticker or RtosTimer.
    * @param sda - I2C Dataline pin
    * @param scl - I2C Clockline pin
    * @param mult - Multiplier with corresponding integration time (M1 = 400ms, M2 = 200ms, M4 = 100ms)
    */
    TSL45315(PinName sda, PinName scl, uint8_t mult);
    
    /** Create a TSL45315 instance without variable integration time ( set to M1) 
    *   and without integrated ticker to repeatedly update the illumination value. 
    *   For use with seperate ticker or RtosTimer.
    * @param sda - I2C Dataline pin
    * @param scl - I2C Clockline pin
    */
    TSL45315(PinName sda, PinName scl);
    
    /** sets the address in the command register of the
    *   target register for future write operations
    * @param reg - specific register to write to (control or config)
    * @param arg - argument of function in the control or config register
    */
    void setReg( int reg, int arg);



    /** reads out datalow and datahigh registers and
    *   calculates the resulting illumination value
    *   in lux which is saved in the variable lux
    * @param reg - specific register to write to
    * @param arg - argument of function in the control or config register
    */
    void getLuxData( );

    /** @returns  returns the actual illumination value in lux which was previously calculated in the getLuxData routine
    */
    uint32_t getLux( ) {
        return lux;
    };

    /** reads out the ID register and saves the id in the variable devID
    */
    void getIDdata( );
    
    /** @returns the ID of the device
    */
    uint8_t getID( ) {
        return devID;
    };
    
    /** reads out datalow and datahigh registers and
    *   calculates the resulting illumination value
    *   in lux which is saved in the variable lux
    * @param mult - Multiplier (1,2,4) defined by M1, M2, M4
    */
    void setMultiplier( uint8_t mult);
    
    /** @returns the multiplier according to the choosen integration time 
    */
    int getMultiplier( ) {
        return multiplier;
    };
    
};
}


/* !_TSL45315_HPP_ */
#endif