// -------------------------------------------------------------------------------------------------------------------
//
//  File: trilateration.h
//
//  Copyright 2016 (c) Decawave Ltd, Dublin, Ireland.
//
//  All rights reserved.
//
//
// * Mbed support added by Harry Nguyen <duongnavy@gmail.com>
// * Porting completed on July/05/2017
// -------------------------------------------------------------------------------------------------------------------
//

#ifndef __TRILATERATION_H__
#define __TRILATERATION_H__


#include "stdio.h"

//#define SHOW_PRINTS

#define TRILATERATION (1)

#define REGRESSION_NUM (10)
#define SPEED_OF_LIGHT      (299702547.0)   // in m/s in air
#define NUM_ANCHORS (5)
#define REF_ANCHOR (5)	//anchor IDs are 1,2,3,4,5 etc. (don't start from 0!)


#define		TRIL_3SPHERES							3
#define		TRIL_4SPHERES							4

typedef struct vec3d	vec3d;
struct vec3d {
double	x;
double	y;
double	z;
};
class Trilateration {

public:
	
	Trilateration();
	vec3d anchor_pos[3];
	vec3d robot_pos;
	int range_array[4];

	
	
	/* Return the difference of two vectors, (vector1 - vector2). */
	vec3d vdiff(const vec3d vector1, const vec3d vector2);
	
	/* Return the sum of two vectors. */
	vec3d vsum(const vec3d vector1, const vec3d vector2);
	
	/* Multiply vector by a number. */
	vec3d vmul(const vec3d vector, const double n);
	
	/* Divide vector by a number. */
	vec3d vdiv(const vec3d vector, const double n);
	
	/* Return the Euclidean norm. */
	double vdist(const vec3d v1, const vec3d v2);
	
	/* Return the Euclidean norm. */
	double vnorm(const vec3d vector);
	
	/* Return the dot product of two vectors. */
	double dot(const vec3d vector1, const vec3d vector2);
	
	/* Replace vector with its cross product with another vector. */
	vec3d cross(const vec3d vector1, const vec3d vector2);
	
	int GetLocation(vec3d *best_solution, int use4thAnchor, vec3d* anchorArray, int *distanceArray);
	
	
	private:
	/* Return the GDOP (Geometric Dilution of Precision) rate between 0-1.
	 * Lower GDOP rate means better precision of intersection.
	 */
	double gdoprate(const vec3d tag, const vec3d p1, const vec3d p2, const vec3d p3);
	
	/* Intersecting a sphere sc with radius of r, with a line p1-p2.
	 * Return zero if successful, negative error otherwise.
	 * mu1 & mu2 are constant to find points of intersection.
	*/
	int sphereline(const vec3d p1, const vec3d p2, const vec3d sc, double r, double *const mu1, double *const mu2);
	
	/* Return TRIL_3SPHERES if it is performed using 3 spheres and return
	 * TRIL_4SPHERES if it is performed using 4 spheres
	 * For TRIL_3SPHERES, there are two solutions: result1 and result2
	 * For TRIL_4SPHERES, there is only one solution: best_solution
	 *
	 * Return negative number for other errors
	 *
	 * To force the function to work with only 3 spheres, provide a duplicate of
	 * any sphere at any place among p1, p2, p3 or p4.
	 *
	 * The last parameter is the largest nonnegative number considered zero;
	 * it is somewhat analogous to machine epsilon (but inclusive).
	*/
	int trilateration(vec3d *const result1,
	                  vec3d *const result2,
	                  vec3d *const best_solution,
	                  const vec3d p1, const double r1,
	                  const vec3d p2, const double r2,
	                  const vec3d p3, const double r3,
	                  const vec3d p4, const double r4,
	                  const double maxzero);
	        
	    /******** END OF FINDING ONE SOLUTION BY INTRODUCING ONE MORE SPHERE *********/
	
	
	/* This function calls trilateration to get the best solution.
	 *
	 * If any three spheres does not produce valid solution,
	 * then each distance is increased to ensure intersection to happens.
	 *
	 * Return the selected trilateration mode between TRIL_3SPHERES or TRIL_4SPHERES
	 * For TRIL_3SPHERES, there are two solutions: solution1 and solution2
	 * For TRIL_4SPHERES, there is only one solution: best_solution
	 *
	 * nosolution_count = the number of failed attempt before intersection is found
	 * by increasing the sphere diameter.
	*/
	int deca_3dlocate ( vec3d   *const solution1,
	                    vec3d   *const solution2,
	                    vec3d   *const best_solution,
	                    int     *const nosolution_count,
	                    double  *const best_3derror,
	                    double  *const best_gdoprate,
	                    vec3d p1, double r1,
	                    vec3d p2, double r2,
	                    vec3d p3, double r3,
	                    vec3d p4, double r4,
	                    int *combination);
	
	
	
	};
#endif

