#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include "MTSText.h"
#include <string>
#include <vector>

using namespace mts;

/* ********************************************** */
/* Create AppEUI identifier and Assigned AppKey   */
/* ********************************************** */

#define USE_OTAA 0 // 1 for OtAA and 0 for Personalized

static const uint8_t DEVKEY[16] = { 0x3D, 0x4B, 0x87, 0x8B, 0x56, 0xEA, 0xA8, 0x1D, 0xA9, 0x7F, 0xB1, 0x9E, 0x69, 0xF8, 0x23, 0x6B }; // Normal
static const uint8_t APPEUI[8]  = { 0xBE, 0x7A, 0x00, 0x00, 0x00, 0x00, 0x02, 0x83  }; // SMTC AppEUI - Most Significant Byte First

//DevAddr  01DC266E 
//NWKSKEY  C7 58 CB 09 BD F2 C7 0E 5F 4C FA 5B 7A 96 68 D8
//APPSKEY  AC B3 B3 A1 B3 26 25 F7 24 3F B4 39 65 06 38 A7
static const uint8_t DEV_ADDR[4] = { 0x01, 0xDC, 0x26, 0x6E };
static const uint8_t NETSKEY[16] = { 0xC7, 0x58, 0xCB, 0x09, 0xBD, 0xF2, 0xC7, 0x0E, 0x5F, 0x4C, 0xFA, 0x5B, 0x7A, 0x96, 0x68, 0xD8 }; // Normal
static const uint8_t APPSKEY[16] = { 0xAC, 0xB3, 0xB3, 0xA1, 0xB3, 0x26, 0x25, 0xF7, 0x24, 0x3F, 0xB4, 0x39, 0x65, 0x06, 0x38, 0xA7 }; // Normal

//DevAddr  01DC266E 
//NWKSKEY  2b 7e 15 16 28 ae d2 a6 ab f7 15 88 09 cf 4f 3c 
//APPSKEY  2b 7e 15 16 28 ae d2 a6 ab f7 15 88 09 cf 4f 3c 
//static const uint8_t DEV_ADDR[4] = { 0x01, 0xDC, 0x26, 0x6E };
//static const uint8_t NETSKEY[16] = { 0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C }; // Normal
//static const uint8_t APPSKEY[16] = { 0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C }; // Normal

static uint8_t       config_frequency_sub_band = 1; // 0 = Enable all channels, 1 = 1st eight channels

/* ********************************************** */

static std::vector<uint8_t> AppKey (DEVKEY, DEVKEY + sizeof(DEVKEY)/sizeof(uint8_t) );
static std::vector<uint8_t> AppEUI (APPEUI, APPEUI + sizeof(APPEUI)/sizeof(uint8_t) ); 

static std::vector<uint8_t> DevAddr (DEV_ADDR, DEV_ADDR + sizeof(DEV_ADDR)/sizeof(uint8_t) );
static std::vector<uint8_t> NetSKey (NETSKEY, NETSKEY + sizeof(NETSKEY)/sizeof(uint8_t) ); 
static std::vector<uint8_t> AppSKey (APPSKEY, APPSKEY + sizeof(APPSKEY)/sizeof(uint8_t) );



InterruptIn button(PB_1);
DigitalOut  drive_high(PB_0);
DigitalIn   door_open(PA_7);

volatile bool msg_rdy;

void queue_message(){
    msg_rdy = true;
}
    

int main() {
    Serial debug(USBTX, USBRX);
    debug.baud(115200);
    drive_high = 1;
    
    msg_rdy = false;
    int32_t ret;
    int32_t next_tx;
    int32_t wait_time = 2;
    mDot* dot;
    std::vector<uint8_t> send_data;
    std::vector<uint8_t> recv_data;
    uint8_t recv = 0;
    uint8_t recv_mismatch = 0;
    uint8_t send_failure = 0;
    uint8_t iterations = 50;
    button.rise(&queue_message);
    
    send_data.push_back(0x01);
    send_data.push_back(0x02);

    dot = mDot::getInstance();

    dot->resetConfig();
    
    dot->setLogLevel(MTSLog::TRACE_LEVEL);
    //dot->setJoinByteOrder(1); // MSB
    dot->setTxDataRate(mDot::SF_10);
    //dot->setTxPower(20);
    
    while ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        logError("failed to set frequency sub band: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    while ((ret = dot->setPublicNetwork(true)) != mDot::MDOT_OK) {
        logError("failed to set Public Network: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    /** Set network ID
     * for use with OTA & AUTO_OTA network join modes
     * setting network ID via this function sets network name to empty
     * @param id a vector of 8 bytes
     * @returns MDOT_OK if success
     */
#if USE_OTAA == 1
    while ((ret = dot->setNetworkId(AppEUI)) != mDot::MDOT_OK) {
        logError("failed to set AppEUI: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    while ((ret = dot->setNetworkKey(AppKey)) != mDot::MDOT_OK) {
        logError("failed to set AppKey: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    dot->setJoinMode( dot->OTA ); // 0 = Manual and 1 = OTA
#else
    while ((ret = dot->setNetworkAddress(DevAddr)) != mDot::MDOT_OK) {
        logError("failed to set DevAddr: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
      while ((ret = dot->setNetworkKey(AppKey)) != mDot::MDOT_OK) {
        logError("failed to set AppKey: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    while ((ret = dot->setNetworkSessionKey(NetSKey)) != mDot::MDOT_OK) {
        logError("failed to set NetworkSessionKey: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    while ((ret = dot->setDataSessionKey(AppSKey)) != mDot::MDOT_OK) {
        logError("failed to set App Session Key: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    dot->setJoinMode( dot->MANUAL ); // 0 = Manual and 1 = OtAA
#endif

    logInfo("enabling activity LED");
    dot->setActivityLedEnable(true);

    logInfo("joining network");
    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
        wait_ms(dot->getNextTxMs() + 1);
    }
    logInfo("joined");

    //dot->setAck(3); // Use Confirmed frames and try three times

    while (1) {

        send_data[0] = 192; //door_open;
        if ((ret = dot->send(send_data)) != mDot::MDOT_OK) {
            logError("failed to send: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
            send_failure++;
        } else {
            logInfo("send data: %s", Text::bin2hexString(send_data).c_str());
            if ((ret = dot->recv(recv_data)) != mDot::MDOT_OK) {
                logError("failed to recv: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
            } else {
                logInfo("recv data: %s", Text::bin2hexString(recv_data).c_str());
                if (recv_data[0] == 0x80) {
                    recv++;
                } else if (recv_data[0] == 0xFF) {
                    goto END;
                } else {
                    recv_mismatch++;
                }
            }
            recv_data.clear();
        }
        msg_rdy = true; // Set to 'true' for free running.
        wait(wait_time);
        while (!msg_rdy) wait(wait_time);
    }
END:    
    logInfo("Version: %s", dot->getId().c_str());
    logInfo("Recv: %d/%d", recv, iterations);
    logInfo("Recv Mismatch: %d/%d", recv_mismatch, iterations);
    logInfo("Send Failure: %d/%d", send_failure, iterations);
    logInfo("Dropped: %d/%d", iterations - (recv + recv_mismatch + send_failure), iterations);

    return 0;
}