#ifndef DDS_H
#define DDS_H

#include "mbed.h"

class DDS {
    
public:
    SPI         _spi;
    DigitalOut  _cs;
    DigitalOut  _rst;
    DigitalOut  update;

    DDS(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName rst, PinName update) :
        _spi(mosi, miso, sclk), _cs(cs), _rst(rst), update(update)
    { 
        // see http://mbed.org/handbook/SPI and page 23 [manual] why format 0
        _spi.format(8, 0);
        _spi.frequency(12e6); // system clock: 1 MHz or 1 us
    };

// ------------------------------------------------------------      
     
     //          *** write and read functions ***
   
// writing to "n_byte" register, with address "address" the value "value"   
    void write(int n_byte, uint32_t address, uint64_t value) {
        update = 0;
        // Instruction byte: (page 25 [manual]) write (0) + internal adress of the register to be written in
        _spi.write(0x00 | (address & 0x1F)); 
        while(n_byte>0) {
            n_byte = n_byte - 1;
           _spi.write((value >> 8*n_byte) & 0xFF);
        }
            update = 1;
            wait(5*1/(12.0e6));
            update =0;
    }    
    
    void ram_write(int n_byte, uint32_t value) {
        while(n_byte>0) {
            n_byte = n_byte - 1;
           _spi.write((value >> 8*n_byte) & 0xFF);
        }
    }    
    
    /*
    void RAM_enable() { 
        int n_byte = 4;
        uint32_t value = 0x80000200;
        _ps0 = 0;
        _spi.write(0x00 | (0x00 & 0x1F));
        while(n_byte>0) {
            n_byte = n_byte - 1;
           _spi.write((value >> 8*n_byte) & 0xFF);
        }
        _ps0 = 1; 
    }*/
    
// Write functions
    void PLSCW_write(uint64_t reg) { write(5, 0x08, reg); }
    void NLSCW_write(uint64_t reg) { write(5, 0x07, reg); }
   // void RSCW0_write(uint64_t reg) { write(5, 0x07, reg); }
    void CFR1_write(uint32_t reg)  { write(4, 0x00, reg); }
    void RAM_write_FTWO(uint32_t reg) { ram_write(4, reg); }
    void RAM_write_PHWO(uint32_t reg) { ram_write(2, reg); }
    void FTW0_write(uint32_t reg) { write(4, 0x04, reg); }
    void FTW1_write(uint32_t reg) { write(4, 0x06, reg); }
    void CFR2_write(uint32_t reg) { write(3, 0x01, reg); }
    void PHWO_write(uint32_t reg) { write(2, 0x05, reg); }
    void ASF_write(uint32_t reg) { write(2, 0x02, reg); }
    void ARR_write(uint32_t reg) { write(1, 0x03, reg); }
    
    
// ------------------------------------------------------------     

// Read 5 byte
    uint64_t read_reg_5byte(uint32_t address) {
        uint64_t value;
        _spi.write(0x80 | (address & 0x0F)); // Instruction byte
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        return value;
    } 
          
// Read 4 byte.
    uint32_t read_reg_4byte(uint32_t address) {
        uint32_t value = 0x00000000;
        _spi.write(0x80 | (address & 0x0F)); // Instruction byte
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        return value;
    }  

// Read 3 byte    
    uint32_t read_reg_3byte(uint32_t address) {
        uint32_t value = 0x000000;
        _spi.write(0x80 | (address & 0x0F)); // Instruction byte
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        return value;
    }   

// Read 2 byte   
    uint32_t read_reg_2byte(uint32_t address) {
        uint32_t value = 0x000000;
        _spi.write(0x80 | (address & 0x0F)); // Instruction byte
        value |= _spi.write(0x00);
        value = value << 8;
        value |= _spi.write(0x00);
        return value;
    } 
    
// Read 1 byte  
    uint32_t read_reg_1byte(uint32_t address) {
        uint32_t value = 0x00;
        _spi.write(0x80 | (address & 0x0F)); // Instruction byte
        value |= _spi.write(0x00);
        return value;
    }   
        
    
// Read functions
    uint64_t RSCW0_read(void) { return read_reg_5byte(0x07); }
    uint64_t PLSCW_read(void) { return read_reg_5byte(0x08); }
    uint64_t NLSCW_read(void) { return read_reg_5byte(0x07); }
    uint32_t CFR1_read(void) { return read_reg_4byte(0x00); }
    uint32_t FTWO_read(void) { return read_reg_4byte(0x04); }  
    uint32_t CFR2_read(void) { return read_reg_3byte(0x01); }
    uint32_t PHWO_read(void) { return read_reg_2byte(0x05); } 
};

#endif