/*
 * Copyright (c) 2017, Aerialspecs, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef ARRAYLIST_H_
#define ARRAYLIST_H_


#include "stdint.h"
#include "Iterator.h"

namespace com {
namespace aerialspecs {
namespace mbed {
namespace sketchlet {

/**
 * \brief Class to represent a simple list of elements in an array.
 *
 * \author Robert R Murrell
 */
template <class T>
class ArrayList {
public:
    /**
     * \brief Creates a new ArrayList with an array size specified by usize.
     *
     * \param usize An unsigned byte specifying the array size, or the maximum number
     *              of elements this array can have.
     */
    ArrayList(uint8_t usize);

    /**
     * \brief Appends an element to the and of the array.
     *
     * This is a template method so elements in an ArrayList will be strongly typed.
     *
     * \param element The element to add to the array.
     *
     * This method will increment the internal index by 1.
     */
    virtual void append(T element);

    virtual T get(uint8_t at);

    /**
     * \brief Returns the number of element in the array.
     *
     * \return internal index + 1.
     */
    virtual uint8_t length();

    /**
     * \brief Returns the maximum number of elements this ArrayList can hold.
     *
     * \return usize.
     */
    virtual uint8_t size();

    /**
     * \brief Creates a new Iterator for this ArrayList.
     *
     * The Iterator is a "point-in-time" iterator in that the iterator can only count to
     * the length at the time of it's creation. An invocation of this method followed by
     * an addition of an element will not be reflected in the Iterator. Iterator is also
     * a template concept and will be strongly typed the same as this ArrayList.
     *
     * \return A new instance of Iterator. The caller must delete this Iterator when complete.
     */
    virtual Iterator<T>* iterate();

    virtual ~ArrayList();

private:
    uint8_t usize;
    uint8_t index;
    T*      elements;
};

}}}}

#endif /* ARRAYLIST_H_ */