/** @headerfile uLCD_4D_Picaso.h
 *
 * This is a class to drive 4D systems screens with Picaso graphics processors
 * 
 * This is a port of the 4D systems Picaso Serial Linux Library
 * Copyright (c) 2014 4D Systems PTY Ltd, Sydney, Australia
 * From https://github.com/4dsystems/Picaso-Serial-Linux-Library, retrieved March 5, 2015
 *
 * Documentation in this file is adapted from the 4D systems Picaso Serial Command Manual, Rev 1.18, Copyright (c) 2014 4D Systems.
 *
 * All software in this library is provided without warranty; use at your own risk.
 */

#ifndef uLCD_4D_Picaso_H
#define	uLCD_4D_Picaso_H

/** @define LCD_USING_MODSERIAL
 *
 * This library can work with either the standard mbed Serial library or with the MODSERIAL
 * library by Andy Kirkham. MODSERIAL allows RX buffering, which could prevent dropped characters
 * at high baud rates. The version of MODSERIAL by Erik Olieman, which is supposed to support more
 * devices, does not appear to work with this library at present, for unknown reasons.
 */
#define LCD_USING_MODSERIAL 1

#include "mbed.h"
#include "Picaso_enums.h"

#if LCD_USING_MODSERIAL
#include "MODSERIAL.h"
#endif



/**
 * @class uLCD_4D_Picaso
 *
 * This class contains all of the Picaso Serial C library functions, ported to mbed.
 */
class uLCD_4D_Picaso {
	public:
	
    /**
     * Class constructor
     *
     * @param tx Serial transmit pin.
     * @param rx Serial receive pin.
     * @param reset Reset GPIO pin.
     */
	uLCD_4D_Picaso(PinName tx, PinName rx, PinName reset);
	
	/**
	 * Resets the LCD screen (takes 3 seconds).
	 */
	void LCD_reset();
	
	
	// LCD text commands:
	
	/**
	 * Prints single character to screen.
	 * @param Character to be printed.
	 */
	void putCH(uint16_t Character);
	
	/**
	 * Prints null-terminated string to screen.
	 * @param InString String to be printed. Maximum length is 511 characters plus null.
	 * @return Number of characters printed.
	 */
	uint16_t putStr(char *InString);
	
	/**
	 * Moves text cursor to specified line and column (based on font size), with (0, 0) being the top-left corner.
	 */
	void txt_MoveCursor(uint16_t Line, uint16_t Column);
	
	/**
	 * Sets text bold, italic, inverse, and underlined attributes.
	 * @param Attribs Word containing attribute flags. OR attribute constants together to set multiple.
	 * @return Previous text attribute word.
	 */
	Picaso::TextAttribute txt_Attributes(Picaso::TextAttribute Attribs);
	
	/**
	 * Sets text inverse attribute.
	 * @return Previous setting.
	 */
	uint16_t txt_Inverse(uint16_t Inverse);
	
	/**
	 * Sets text italic attribute.
	 * @return Previous setting.
	 */
	uint16_t txt_Italic(uint16_t Italic);
	
	/**
	 * Sets text bold attribute.
	 * @return Previous setting.
	 */
	uint16_t txt_Bold(uint16_t Bold);
	
	/**
	 * Sets text underline attribute.
	 * @return Previous setting.
	 */
	uint16_t txt_Underline(uint16_t Underline);
	
	/**
	 * Sets text background color.
	 * @return Previous color setting.
	 */
	Picaso::Color txt_BGcolour(Picaso::Color Color);
	
	/*
	 * Sets text foreground color.
	 * @return Previous color setting.
	 */
	Picaso::Color txt_FGcolour(Picaso::Color Color);
	
	/**
	 * Sets text background opacity (i.e. whether background pixels are drawn).
	 * @return Previous setting.
	 */
	Picaso::TextOpacity txt_Opacity(Picaso::TextOpacity TransparentOpaque);
	
	/**
	 * Sets text font. Note that one can also pass a handle to a file with a font: see Picaso Serial Command Manual for more information.
	 * @return Previous setting.
	 */
	Picaso::Font txt_FontID(Picaso::Font FontNumber);
	
	/**
	 * Sets text font height.
	 * @param Multiplier Font height as multiple of default height.
	 * @return Previous setting.
	 */
	uint16_t txt_Height(uint16_t Multiplier);
	
	/**
	 * Sets text font width.
	 * @param Multiplier Font width as multiple of default.
	 * @return Previous setting.
	 */
	uint16_t txt_Width(uint16_t Multiplier);
	
	/**
	 * Returns the height, in pixels, of a given character under the current font settings. If height is greater than 255, the returned value will be modulo 8.
	 */
	uint16_t charheight(char  TestChar);
	
	/**
	 * Returns the width, in pixels, of a given character under the current font settings. If width is greater than 255, the returned value will be modulo 8.
	 */
	uint16_t charwidth(char  TestChar);
	
	/**
	 * Sets the pixel position (from left side) at which text wraps around on right side.
	 * @return Previous setting.
	 */
	uint16_t txt_Wrap(uint16_t  Position);
	
	/**
	 * Sets gap, in pixels, between characters in the x-direction.
	 * @return Previous setting.
	 */
	uint16_t txt_Xgap(uint16_t  Pixels);
	
	/**
	 * Sets gap, in pixels, between characters in the y-direction.
	 * @return Previous setting.
	 */
	uint16_t txt_Ygap(uint16_t  Pixels);
	
	/**
	 * Sets one of several text attributes. Most attributes can be set by other functions.
	 * @param Func Attribute to set.
	 * @param Value Value to set parameter. See TextSetFunc definition for more information.
	 */
	void txt_Set(Picaso::TextSetFunc Func, uint16_t Value);
	
	
	// LCD graphics commands:
	
	/**
	 * Clears screen and sets several attributes (such as transparency, outline color, opacity, pen style, line patterning, 
	 * text wrap position, text size, and origin) to their default values.
	 */
	void gfx_Cls(void);
	
	/**
	 * Changes all pixels on the screen of OldColor to NewColor.
	 */
	void gfx_ChangeColour(uint16_t  OldColor, uint16_t  NewColor);
	
	/**
	 * Draws pixel of color Color at (X, Y).
	 */
	void gfx_PutPixel(uint16_t  X, uint16_t  Y, Picaso::Color Color);
	
	/**
	 * Returns current color of pixel at (X, Y).
	 */
	Picaso::Color gfx_GetPixel(uint16_t  X, uint16_t  Y);
	
	/**
	 * Draws (unfilled) circle on screen.
	 * @param X,Y Coordinates (in pixels) of circle's center.
	 * @param Radius Radius of circle.
	 * @param Color Color of circle.
	 */
	void gfx_Circle(uint16_t  X, uint16_t  Y, uint16_t  Radius, Picaso::Color Color);
	
	/**
	 * Draws filled circle on screen.
	 * @param X,Y Coordinates (in pixels) of circle's center.
	 * @param Radius Radius of circle.
	 * @param Color Color of circle.
	 */
	void gfx_CircleFilled(uint16_t  X, uint16_t  Y, uint16_t  Radius, Picaso::Color  Color);
	
	/**
	 * Draws line from (X1, Y1) to (X2, Y2) with color Color
	 */
	void gfx_Line(uint16_t X1, uint16_t Y1, uint16_t X2, uint16_t Y2, Picaso::Color Color);
	
	/**
	 * Draws unfilled rectangle on screen.
	 * @param X1,Y1 Coordinates of top-left corner of rectangle.
	 * @param X2,Y2 Coordinates of bottom-right corner of rectangle.
	 * @param Color of rectangle.
	 */
	void gfx_Rectangle(uint16_t  X1, uint16_t  Y1, uint16_t  X2, uint16_t  Y2, Picaso::Color Color);
	
	/**
	 * Draws filled rectangle on screen.
	 * @param X1,Y1 Coordinates of top-left corner of rectangle.
	 * @param X2,Y2 Coordinates of bottom-right corner of rectangle.
	 * @param Color of rectangle.
	 */
	void gfx_RectangleFilled(uint16_t  X1, uint16_t  Y1, uint16_t  X2, uint16_t  Y2, Picaso::Color Color);
	
	/**
	 * Plots line between a sequence of points.
	 * @param n Number of vertices.
	 * @param Xvalues,Yvalues Arrays of size n containing the coordinates of each vertex.
	 * @param Color Color of line.
	 */
	void gfx_Polyline(uint16_t  n, uint16_t*  Xvalues, uint16_t*  Yvalues, Picaso::Color Color);
	
	/**
	 * Draws unfilled polygon on screen.
	 * @param n Number of vertices.
	 * @param Xvalues,Yvalues Arrays of size n contianing the coordinates of each vertex.
	 * @param Color Color of polygon.
	 */
	void gfx_Polygon(uint16_t  n, uint16_t*  Xvalues, uint16_t*  Yvalues, Picaso::Color Color);
	
	/**
	 * Draws filled polygon on screen.
	 * @param n Number of vertices.
	 * @param Xvalues,Yvalues Arrays of size n contianing the coordinates of each vertex.
	 * @param Color Color of polygon.
	 */
	void gfx_PolygonFilled(uint16_t  n, uint16_t*  Xvalues, uint16_t*  Yvalues, Picaso::Color Color);
	
	/**
	 * Draws unfilled triangle on screen.
	 * @param X1,Y1 Coordinates of first vertex.
	 * @param X2,Y2 Coordinates of second vertex.
	 * @param X3,Y3 Coordinates of third vertex.
	 * @param Color Color of triangle.
	 */
	void gfx_Triangle(uint16_t  X1, uint16_t  Y1, uint16_t  X2, uint16_t  Y2, uint16_t  X3, uint16_t  Y3, Picaso::Color Color);
	
	/**
	 * Draws filled triangle on screen.
	 * @param X1,Y1 Coordinates of first vertex.
	 * @param X2,Y2 Coordinates of second vertex.
	 * @param X3,Y3 Coordinates of third vertex.
	 * @param Color Color of triangle.
	 */
	void gfx_TriangleFilled(uint16_t  X1, uint16_t  Y1, uint16_t  X2, uint16_t  Y2, uint16_t  X3, uint16_t  Y3, Picaso::Color Color);
	
	/**
	 * Draws ellipse on screen.
	 * @param X,Y Coordinates of center of ellipse.
	 * @param Xrad Ellipse's x-radius.
	 * @param Yrad Ellipse's y-radius.
	 * @param Color Color of ellipse.
	 */
	void gfx_Ellipse(uint16_t X, uint16_t Y, uint16_t Xrad, uint16_t Yrad, Picaso::Color Color);
	
	/**
	 * Draws filled ellipse on screen.
	 * @param X,Y Coordinates of center of ellipse.
	 * @param Xrad Ellipse's x-radius.
	 * @param Yrad Ellipse's y-radius.
	 * @param Color Color of ellipse.
	 */
	void gfx_EllipseFilled(uint16_t X, uint16_t Y, uint16_t Xrad, uint16_t Yrad, Picaso::Color Color);
	
	
	
	
	/**
	 * Enables or disables clipping.
	 * @param OnOff 0 = Off, 1 = On.
	 */
	void gfx_Clipping(uint16_t  OnOff);
	
	/**
	 * Sets clipping window for graphics. Any objects and text will only be displayed within the window.
	 * Clipping must first be enabled with the gfx_Clipping command.
	 * @param X1, Y1 Coordinates of top left corner of window.
	 * @param X2, Y2 Coordinates of bottom right corner of window.
	 */
	void gfx_ClipWindow(uint16_t  X1, uint16_t  Y1, uint16_t  X2, uint16_t  Y2);
	
	/**
	 * Forces clip window to include the last printed text or image.
	 */
	void gfx_SetClipRegion(void);
	
	
	
	
	
	/**
	 * Draws line from the origin to (X, Y), using current object color and then moves the origin to the endpoint. This command is most useful when used with gfx_Orbit.
	 */
	void gfx_LineTo(uint16_t  X, uint16_t  Y);
	
	/**
	 * Moves the origin to (X, Y).
	 */
	void gfx_MoveTo(uint16_t  X, uint16_t  Y);
	
	/**
	 * Calculates the X and Y coordinates relative to the origin, given an angle and distance.
	 * @param Angle Angle in degrees.
	 * @param Distance Distance from origin in pixels.
	 * @param Xdest Pointer to variable where calculated X value will be stored.
	 * @param Ydest Pointer to variable where calculated Y value will be stored.
	 * @return Returns 0.
	 */
	uint16_t gfx_Orbit(uint16_t Angle, uint16_t Distance, uint16_t* Xdest, uint16_t* Ydest);
	
	
	/**
	 * Draws button on screen.
	 * @param Up Whether button appears raised (BUTTON_RAISED = 1) or depressed (BUTTON_DEPRESSED = 0).
	 * @param x,y Coordinates of top left corner of button.
	 * @param buttonColour Button color.
	 * @param txtColour Text color.
	 * @param font Text font ID.
	 * @param txtWidth Text width, as multiple of default.
	 * @param txtHeight Text height, as multiple of default.
	 * @param text Null-terminated string with button text. May include newline characters for multiple lines of text in button.
	 */
	void gfx_Button(Picaso::ButtonState Up, uint16_t x, uint16_t y, Picaso::Color buttonColour,
			Picaso::Color txtColour, Picaso::Font font, uint16_t txtWidth, uint16_t txtHeight, char* text);
	
	/**
	 * 
	 * @param Raised Panel appearance (raised or sunken).
	 * @param X,Y Position of top-left corner of panel.
	 * @param Width Panel width.
	 * @param Height Panel height.
	 * @param Color Panel color.
	 */
	void gfx_Panel(Picaso::PanelState Raised, uint16_t X, uint16_t Y, uint16_t Width, uint16_t  Height, Picaso::Color Color);
	
	/**
	 * Draws horizontal or vertical slider bar on screen. If X2 - X1 > Y2 - Y1, slider is assumed to be horizontal. Otherwise, slider is assumed to be vertical.
	 * @param Mode Slider appearance (sunken, raised, or hidden).
	 * @param X1,Y1 Coordinates of top-left corner of slider.
	 * @param X2,Y2 Coordinates of bottom-right corner of slider.
	 * @param Color Color of slider bar.
	 * @param Scale Sets scale of slider position: slider can have positions from 0 to Scale.
	 * @param Position of slider thumb. If negative, the absolute value is taken.
	 * @return Unknown--Serial command manual does not specify a return.
	 */
	uint16_t gfx_Slider(Picaso::SliderState Mode, uint16_t  X1, uint16_t  Y1, uint16_t  X2, uint16_t  Y2, Picaso::Color Color, uint16_t  Scale, uint16_t  Value);
	
	/**
	 * Sets depth of bevel shadow on buttons drawn.
	 * @param Value Shadow depth in pixels (0-4).
	 * @return Previous setting.
	 */
	uint16_t gfx_BevelShadow(uint16_t  Value);
	
	/**
	 * Sets width of bevel on buttons drawn.
	 * @param Value Shadow depth in pixels (0-15).
	 * @return Previous setting.
	 */
	uint16_t gfx_BevelWidth(uint16_t  Value);
	
	/**
	 * Copies a region of the screen and pastes it to another.
	 * @param Xs,Ys Coordinates of top-left corner of region to be copied.
	 * @param Xd,Yd Coordinates of top-left corner of destination region.
	 * @param Width Width of region to be copied.
	 * @param Height Height of region to be copied.
	 */
	void gfx_ScreenCopyPaste(uint16_t  Xs, uint16_t  Ys, uint16_t  Xd, uint16_t  Yd, uint16_t  Width, uint16_t  Height);
	
	/**
	 * Sets the screen background color.
	 * @return Previous background color.
	 */
	Picaso::Color gfx_BGcolour(Picaso::Color Color);
	
	
	/**
	 * Sets the outline color for rectangles and circles.
	 * @return Previous outline color.
	 */
	Picaso::Color gfx_OutlineColour(Picaso::Color Color);
	
	/**
	 * Sets display contrast or turns it on and off, depending on display model.
	 * @param Contrast Constrast setting. For most displays, 0 = display off, non-zero = display on. See Serial Command Manual for exceptions.
	 * @return Previous setting.
	 */
	uint16_t gfx_Contrast(uint16_t  Contrast);
	
	/**
	 * Sets inter-frame delay, in milliseconds, for media_Video command.
	 * @return Previous setting.
	 */
	uint16_t gfx_FrameDelay(uint16_t  Msec);
	
	/**
	 * Sets line drawing pattern.
	 * @param Pattern Word containing bit pattern; each set bit corresponds to a pixel turned off.
	 * @return Previous pattern.
	 */
	uint16_t gfx_LinePattern(uint16_t  Pattern);
	
	/**
	 * Sets screen mode (portait, landscape, and reversed).
	 * @return Previous screen mode.
	 */
	Picaso::ScreenMode gfx_ScreenMode(Picaso::ScreenMode ScreenMode);
	
	/**
	 * Turns transparency ON or OFF. Setting is reset to OFF after image or video command.
	 * @return Previous setting.
	 */
	uint16_t gfx_Transparency(uint16_t OnOff);
	
	/**
	 * Sets color to be made transparent.
	 * @return Previous setting.
	 */
	Picaso::Color gfx_TransparentColour(Picaso::Color Color);
	
	/**
	 * Sets one of several graphics attributes. Most attributes can be set by other functions.
	 * @param Func Attribute to set.
	 * @param Value Value to set parameter. See GFXSetFunc definition for more information.
	 */
	void gfx_Set(Picaso::GFXSetFunc Func, uint16_t Value);
	
	/**
	 * Returns a graphics attribute.
	 * @return Value of attribute. See GFXGetMode definition for more information.
	 */
	uint16_t gfx_Get(Picaso::GFXGetMode Mode);
	
	
	// Media (SD card) commands:
	
	/**
	 * Initializes SD card.
	 * @return 1 if successful, 0 otherwise.
	 */
	uint16_t media_Init(void);
	
	/**
	 * Sets media memory address to a non-sector-aligned byte address in the SD card.
	 * @param HiWord Upper 2 bytes of address.
	 * @param LoWord Lower 2 bytes of address.
	 */
	void media_SetAdd(uint16_t  HiWord, uint16_t  LoWord);
	
	/**
	 * Sets media memory address to a sector in the SD card.
	 * @param HiWord Upper 2 bytes of address.
	 * @param LoWord Lower 2 bytes of address.
	 */
	void media_SetSector(uint16_t  HiWord, uint16_t  LoWord);
	
	/**
	 * Reads sector of SD card at address previously set by media_SetSector. Afterwards, the sector address is incremented by one.
	 * @param SectorIn Array to hold sector data (512 bytes).
	 * @return 1 if successful, 0 if failed.
	 */
	uint16_t media_RdSector(char* SectorIn);
	
	/**
	 * Writes sector of SD card at address previously set by media_SetSector. Afterwards, the sector address is incremented by one.
	 * @param SectorOut Array with sector data to write (512 bytes).
	 * @return 1 if successful, 0 if failed.
	 */
	uint16_t media_WrSector(char* SectorOut);
	
	/**
	 * Reads byte from SD card at address previously set by media_SetAdd. Afterwards, the byte address is incremented by one.
	 * @return Byte read (lower 8 bits of word returned).
	 */
	uint16_t media_ReadByte(void);
	
	/**
	 * Writes byte to SD card at address previously set by media_SetAdd. Afterwards, the byte address is incremented by one.
	 * @param Byte to write (in lower 8 bits of argument).
	 * @return Nonzero if successful, 0 if failed.
	 */
	uint16_t media_WriteByte(uint16_t  Byte);
	
	/**
	 * Reads word from SD card at address previously set by media_SetAdd. Afterwards, the byte address is incremented by one.
	 * @return Word read.
	 */
	uint16_t media_ReadWord(void);
	
	/**
	 * Writes byte to SD card at address previously set by media_SetAdd. Afterwards, the byte address is incremented by one.
	 * @param Word to write.
	 * @return Nonzero if successful, 0 if failed.
	 */
	uint16_t media_WriteWord(uint16_t  Word);
	
	/**
	 * Finalizes writing to sector. This is automatically called if writing rolls over into the next sector.
	 * @return Nonzero if successful, 0 of failed.
	 */
	uint16_t media_Flush(void);
	
	/**
	 * Displays RAW image from SD card at at the address specified by one of the set address commands.
	 * @param X,Y Coordinates of top left position of image on screen.
	 */
	void media_Image(uint16_t  X, uint16_t  Y);
	
	/**
	 * Displays RAW video clip from SD card at at the address specified by one of the set address commands. All other processes on display are blocked until video is finished.
	 * @param X,Y Coordinates of top left position of video on screen.
	 */
	void media_Video(uint16_t  X, uint16_t  Y);
	
	/**
	 * Displays individual frame of RAW video clip from SD card at at the address specified by one of the set address commands.
	 * @param X,Y Coordinates of top left position of frame on screen.
	 * @param Framenumber Number of frame to be displayed.
	 */
	void media_VideoFrame(uint16_t  X, uint16_t  Y, uint16_t  Framenumber);
	
	
	// File commands:
	
	/**
	 * Returns most recent file operation error code.
	 */
	Picaso::FileError file_Error(void);
	
	/**
	 * Returns the number of files that match the given (null-terminated) Filename string. In the string, '*' can represent any combination of allowable characters,
	 * while '?' matches any single allowable character. Filenames must be 8.3 format.
	 */
	uint16_t file_Count(char * Filename);
	
	/**
	 * Prints on the screen the names of files that match the given (null-terminated) Filename string. In the string, '*' can represent any combination of allowable characters,
	 * while '?' matches any single allowable character. Filenames must be 8.3 format.
	 * @return Number of matching files found.
	 */
	uint16_t file_Dir(char * Filename);
	
	/**
	 * Prints on the screen the name of the first file that matches the given (null-terminated) Filename string. In the string, '*' can
	 * represent any combination of allowable characters, while '?' matches any single allowable character. Filenames must be 8.3 format.
	 * @return 1 if at least one matching file is found, 0 otherwise.
	 */
	uint16_t file_FindFirst(char *Filename);
	
	/**
	 * Finds the name of the first file that matches the given (null-terminated) Filename string. In the string, '*' can
	 * represent any combination of allowable characters, while '?' matches any single allowable character. Filenames must be 8.3 format.
	 * @param Filename String containing filename pattern to be matched.
	 * @param StringIn Character buffer to hold the returned filename. Returned string is not null-terminated.
	 * @return Length of returned filename string.
	 */
	uint16_t file_FindFirstRet(char *Filename, char* StringIn);
	
	/**
	 * Prints on the screen the name of the next file to match the pattern provided in a previous call to file_FindFirst or file_FindFirstRet.
	 * @return 1 if at least one matching file is found, 0 otherwise.
	 */
	uint16_t file_FindNext(void);
	
	/**
	 * Finds the name of the next file to match the pattern provided in a previous call to file_FindFirst or file_FindFirstRet.
	 * @param StringIn Character buffer to hold the returned filename. Returned string is not null-terminated.
	 * @return Length of returned filename string.
	 */
	uint16_t file_FindNextRet(char * StringIn);
	
	/**
	 * Returns whether a file with a given (null-terminated) filename exists.
	 * @return 1 if found, 0 if not.
	 */
	uint16_t file_Exists(char *Filename);
	
	/**
	 * Opens file.
	 * @param Filename Null-terminated string with name of file.
	 * @param Mode 'r' for read, 'w' for write, 'a' for append.
	 * @return File handle (if file exists).
	 */
	uint16_t file_Open(char * Filename, char  Mode);
	
	/**
	 * Closes file.
	 * @param Handle File handle.
	 * @return 1 if successfully closed, 0 if not.
	 */
	uint16_t file_Close(uint16_t  Handle);
	
	/**
	 * Reads bytes from file.
	 * @param Data Buffer to contain data read.
	 * @param Size Number of bytes to read.
	 * @param Handle File handle.
	 * @return Number of bytes read.
	 */
	uint16_t file_Read(char* Data, uint16_t Size, uint16_t Handle);
	
	/**
	 * Sets internal file pointer to a specified position in a file.
	 * @param Handle File handle.
	 * @param HiWord Upper 16 bits of pointer.
	 * @param LoWord Lower 16 bits of pointer.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t file_Seek(uint16_t  Handle, uint16_t  HiWord, uint16_t  LoWord);
	
	/**
	 * Sets file pointer to a particular record within a file (e.g. a record size of 1000 and an index of 9 will set the pointer to position 9000).
	 * @param Handle File handle.
	 * @param HiSize Upper 16 bits of record size.
	 * @param LoSize Lower 16 bits of record size.
	 * @param Recordnum Index of desired record.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t file_Index(uint16_t  Handle, uint16_t  HiSize, uint16_t  LoSize, uint16_t  Recordnum);
	
	/**
	 * Returns the current position of a file pointer.
	 * @param Handle File handle.
	 * @param HiWord Pointer to word in which to store the upper 16 bits of the returned pointer value.
	 * @param LoWord Pointer to word in which to store the lower 16 bits of the returned pointer value.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t file_Tell(uint16_t  Handle, uint16_t *  HiWord, uint16_t *  LoWord);
	
	/**
	 * Writes bytes to file.
	 * @param Size Number of bytes to be written.
	 * @param Source Array containing data to be written.
	 * @param Handle File handle.
	 * @return Number of bytes written.
	 */
	uint16_t file_Write(uint16_t  Size, char*  Source, uint16_t  Handle);
	
	/**
	 * Returns the size of a file.
	 * @param Handle File handle.
	 * @param HiWord Pointer to variable in which to store the upper 16 bits of the returned file size.
	 * @param LoWord Pointer to variable in which to store the lower 16 bits of the returned file size.
	 * @return 1 if succesful, 0 if not.
	 */
	uint16_t file_Size(uint16_t  Handle, uint16_t *  HiWord, uint16_t *  LoWord);
	
	/**
	 * Displays image from file on screen. If the file contains more than one image, each can be accessed with the file_Seek command.
	 * @param X,Y coordinates of top-left corner of image on display.
	 * @param Handle File handle.
	 * @return File error code (see file_Error).
	 */
	uint16_t file_Image(uint16_t  X, uint16_t  Y, uint16_t  Handle);
	
	/**
	 * Captures an area of the screen and writes it to a file.
	 * @param X,Y Coordinates of top-left corner of area to be captured.
	 * @param Width, Height Width and height of area to be captured.
	 * @param Handle Handle of file to be written. Image is written at the current pointer location in the file, so multiple images can be captured to the same file.
	 * @return 0 if successful.
	 */
	uint16_t file_ScreenCapture(uint16_t  X, uint16_t  Y, uint16_t  Width, uint16_t  Height, uint16_t  Handle);
	
	/**
	 * Writes character to file.
	 * @param Character Character to be written.
	 * @param Handle File handle.
	 * @return Number of bytes successfully written.
	 */
	uint16_t file_PutC(char  Character, uint16_t  Handle);
	
	/**
	 * Reads character from file.
	 */
	char file_GetC(uint16_t  Handle);
	
	/**
	 * Writes word (2 bytes) to file.
	 * @return Number of bytes successfully written.
	 */
	uint16_t file_PutW(uint16_t  Word, uint16_t  Handle);
	
	/**
	 * Reads word from file.
	 */
	uint16_t file_GetW(uint16_t  Handle);
	
	/**
	 * Writes (null-terminated) string to file.
	 * @return Number of characters written (excluding null character).
	 */
	uint16_t file_PutS(char * StringOut, uint16_t  Handle);
	
	/**
	 * Reads line of text from file. Characters are read until a newline or eof is reached, or the specified maximum size is reached. Returned string is not null-terminated.
	 * @param StringIn Character array in which to store read string.
	 * @param Size maximum number of characters to be read.
	 * @param Handle File handle.
	 * @return Number of characters read.
	 */
	uint16_t file_GetS(char * StringIn, uint16_t  Size, uint16_t  Handle);
	
	/**
	 * Deletes file from disk.
	 * @param Filename Name of file to be erased.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t file_Erase(char * Filename);
	
	/**
	 * Resets file pointer to beginning of file.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t file_Rewind(uint16_t  Handle);
	
	/**
	 * Loads 4DGL function (.4FN) or program (.4XE) from file to memory.
	 * @return Pointer/handle to loaded function.
	 */
	uint16_t file_LoadFunction(char *Filename);
	
	/**
	 * Calls a previously-loaded function or program.
	 * @param Handle Function handle previously returned by file_LoadFunction.
	 * @param ArgCount Number of arguments passed to function (0-6).
	 * @param Args array containing arguments to be passed to function.
	 * @return Value returned by the "main" function.
	 */
	uint16_t file_CallFunction(uint16_t  Handle, uint16_t  ArgCount, uint16_t*  Args);
	
	/**
	 * Runs a 4DGL program from a file. This is similar to file_Execute, but afterwards, all memory allocations are released (but not stack and global memory.
	 * Prior to execution, strings may be loaded with the writeString command.
	 * @param Filename Null-terminated string containing name of file containing program.
	 * @param ArgCount Number of arguments to be passed to program.
	 * @param Args Array of arguments to be passed to program.
	 * @return Value returned by program.
	 */
	uint16_t file_Run(char *Filename, uint16_t  ArgCount, uint16_t*  Args);
	
	/**
	 * Runs a 4DGL program from a file. This is similar to file_Run, except all memory allocations are retained.
	 * Prior to execution, strings may be loaded with the writeString command.
	 * @param Filename Null-terminated string containing name of file containing program.
	 * @param ArgCount Number of arguments to be passed to program.
	 * @param Args Array of arguments to be passed to program.
	 * @return Value returned by program.
	 */
	uint16_t file_Exec(char *Filename, uint16_t  ArgCount, uint16_t*  Args);
	
	/**
	 * Loads image control file and associated image file. See Serial Command Manual for more information.
	 * @param Datname String containing control list filename (.dat).
	 * @param GCIName String containing image filename (.gci).
	 * @param Mode Mode of operation (0-2).
	 */
	uint16_t file_LoadImageControl(char *Datname, char *GCIName, uint16_t  Mode);
	
	/**
	 * Mounts the FAT file system. This must be called before any other file commands.
	 * @return Non-zero if successful, 0 otherwise.
	 */
	uint16_t file_Mount(void);
	
	/**
	 * Unmounts the FAT file system.
	 */
	void file_Unmount(void);
	
	/**
	 * Plays .wav audio file.
	 * @return If successful, number of blocks to play. If not, an error code (see command manual).
	 */
	uint16_t file_PlayWAV(char * Filename);
	
	/**
	 * Loads string to memory to be used by a 4DGL program/function.
	 * @param Handle Pointer to location where string is to be loaded. Initial call should used 0; each subsequent call should use the pointer returned by the previous call.
	 * @param StringOut Null-terminated string to be loaded.
	 * @return Pointer to location where string has been loaded.
	 */
	uint16_t writeString(uint16_t  Handle, char * StringOut);
	
	/**
	 * Reads string which has been written by a 4DGL program/function. The location is initially written with the writeString command, the pointer is passed to the program
	 * (which can modify it), and the result is read with this function.
	 * @param Handle Pointer to string location. This is the same value passed to writeString when the space was initially written.
	 * @param StringIn Character array in which to store the returned string.
	 * @return Unknown--command manual does not specify a return.
	 */
	uint16_t readString(uint16_t  Handle, char * StringIn);
	
	
	// Sound playback commands:
	
	/**
	 * Sets sound playback volume.
	 * @param Volume Volume setting (8-127). Any level outside the allowed range will be changed to the closest value within the range.
	 */
	void snd_Volume(uint16_t  Volume);
	
	/**
	 * Sets playback sample rate.
	 * @param Pitch Sample rate (4000-65535). A value of 0 will restore the original sample rate.
	 */
	uint16_t snd_Pitch(uint16_t  Pitch);
	
	/**
	 * Sets memory chunk size for wavefile buffer.
	 * @param Bufsize Buffer size: 0 -> 1024 bytes, 1 -> 2048 bytes, and 2 -> 4096 bytes.
	 */
	void snd_BufSize(uint16_t  Bufsize);
	
	/**
	 * Stops sound that is playing, releasing buffers and closing open wav files.
	 */
	void snd_Stop(void);
	
	/**
	 * Pauses sound playback.
	 */
	void snd_Pause(void);
	
	/**
	 * Continues any paused sound.
	 */
	void snd_Continue(void);
	
	/**
	 * Returns 0 if file has finished playing; otherwise, returns number of 512 byte blocks remaining.
	 */
	uint16_t snd_Playing(void);
	
	
	/**
	 * Specifies region on screen where touches will be detected.
	 * @param X1,Y1 Coordinates of top-left corner of region.
	 * @param X2,Y2 Coordinates of bottom-right corner of region.
	 */
	void touch_DetectRegion(uint16_t  X1, uint16_t  Y1, uint16_t  X2, uint16_t  Y2);
	
	/**
	 * Sets touchscreen mode.
	 * @param Mode 0 -> enable touch screen; 1 -> disable touch screen; 2 -> reset touch detect region to full screen. The Picaso::Touch enum also provides the constants.
	 */
	void touch_Set(uint16_t  Mode);
	
	/**
	 * Get status of touch screen.
	 * @param Mode 0 -> get state of touch screen; 1 -> get x coordinate of touch; 2 -> get y coordinate of touch. The Picaso::Touch enum also provides the constants.
	 * @return In modes 1 and 2, the requested coordinate. In mode 0: 0 -> invalid or no touch; 1 -> press; 2 -> release; 3 -> moving. The Picaso::Touch enum also provides the constants.
	 */
	uint16_t touch_Get(uint16_t  Mode);
	
	/**
	 * Sets position of image to be displayed. An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list.
	 * @param Xpos,Ypos Coordinates of top-left corner of image.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t img_SetPosition(uint16_t  Handle, uint16_t  Index, uint16_t  Xpos, uint16_t  Ypos);
	
	/**
	 * Enables image in list so that it can be displayed with the img_Show command. An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list. -1 or Picaso::ALL selects all images in list.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t img_Enable(uint16_t  Handle, uint16_t  Index);
	
	/**
	 * Disables image from being displayed. An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list. -1 or Picaso::ALL selects all images in list.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t img_Disable(uint16_t  Handle, uint16_t  Index);
	
	/**
	 * Darkens image (call before img_Show). This effect will be reset when img_Show is called a second time.
	 * An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list. -1 or Picaso::ALL selects all images in list.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t img_Darken(uint16_t  Handle, uint16_t  Index);
	
	/**
	 * Lightens image (call before img_Show). This effect will be reset when img_Show is called a second time.
	 * An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list. -1 or Picaso::ALL selects all images in list.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t img_Lighten(uint16_t  Handle, uint16_t  Index);
	
	/**
	 * Lightens image (call before img_Show). This effect will be reset when img_Show is called a second time.
	 * An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list.
	 * @param Offset Offset of word to write. Note that some parameters are read-only.
	 * @param Word Word to be written. See Picaso::ImageControlOffset definition or serial command manual for more information.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t img_SetWord(uint16_t  Handle, uint16_t  Index, Picaso::ImageControlOffset Offset, uint16_t  Word);
	
	/**
	 * Reads an image parameter. An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list.
	 * @param Offset Offset of word to be read.
	 * @return The word that is read.
	 */
	uint16_t img_GetWord(uint16_t  Handle, uint16_t  Index, Picaso::ImageControlOffset Offset);
	
	/**
	 * Displays image from list. An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list.
	 * @return Nonzero if successful, 0 if not.
	 */
	uint16_t img_Show(uint16_t  Handle, uint16_t  Index);
	
	/**
	 * Sets one or more image attribute flags. An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list.
	 * @param Value Mask of bits to be set. OR together multiple attribute flags to set multiple.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t img_SetAttributes(uint16_t  Handle, uint16_t  Index, Picaso::ImageAttribute Value);
	
	/**
	 * Clears one or more image attribute flags. An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list.
	 * @param Value Mask of bits to be cleared. OR together multiple attribute flags to clear multiple.
	 * @return 1 if successful, 0 if not.
	 */
	uint16_t img_ClearAttributes(uint16_t  Handle, uint16_t  Index, Picaso::ImageAttribute Value);
	
	/**
	 * Returns whether an image has been touched. An image control must have already been created with the file_LoadImageControl command.
	 * @param Handle Pointer to image list.
	 * @param Index Index of image in list.
	 * @return Image index if touched, -1 if not. If -1 is passed as index, returns index of an image if it is touched.
	 */
	uint16_t img_Touched(uint16_t  Handle, uint16_t  Index);
	
	/**
	 * Performs block image transfer onto screen.
	 * @param X,Y Coordinates of top-left corner of location where image is to be displayed.
	 * @param Width,Height Width and height of image to be displayed.
	 * @param Pixels Array of bytes containing pixel data. Each pixel is 16-bits of color information (see Picaso::Color).
	 */
	void blitComtoDisplay(uint16_t  X, uint16_t  Y, uint16_t  Width, uint16_t  Height, char*  Pixels);
	
	
	// System commands:
	
	/**
	 * Releases memory block.
	 * @param Handle Pointer to memory block.
	 * @return Nonzero if successful, 0 if not.
	 */
	uint16_t mem_Free(uint16_t  Handle);
	
	/**
	 * Returns byte size of lergest available memory chunk on the heap.
	 */
	uint16_t mem_Heap(void);
	
	/**
	 * Returns contents of memory address. Addresses can be calculated with pointers and offsets, and Picaso::MemAddresses and Picaso::MemOffsets also have some useful values.
	 */
	uint16_t peekM(uint16_t  Address) ;
	
	/**
	 * Writes word to memory address. Addresses can be calculated with pointers and offsets, and Picaso::MemAddresses and Picaso::MemOffsets also have some useful values.
	 */
	void pokeM(uint16_t  Address, uint16_t  WordValue) ;
	
	/**
	 * Returns display model name as unterminated string.
	 * @param ModelStr Array of characters in which to store returned string.
	 * @return Number of characters in returned string.
	 */
	uint16_t sys_GetModel(char *ModelStr);
	
	/**
	 * Returns version of the SPE (Serial Programming Environment) installed on the display.
	 */
	uint16_t sys_GetVersion(void);
	
	/**
	 * Returns version of the PmmC (firmware) installed on the display.
	 */
	uint16_t sys_GetPmmC(void);
	
	/**
	 * Puts display and processor in low-power state for specified period of time.
	 * @param Units If 1-65535, sleeps for that many units of time (each unit is approximately 1 second). If 0, sleeps forever (needs to be reset).
	 */
	uint16_t sys_Sleep(uint16_t  Units);
	
	
	// I/O commands:
	
	/**
	 * Returns value of bus pins (in lower 8 bits of returned value).
	 */
	uint16_t bus_In(void);
	
	/**
	 * Sets value of bus pins. Bus pins must first be set to output.
	 * @param Bits Value to write to pins in lower 8 bits. Upper 8 bits are ignored.
	 */
	void bus_Out(uint16_t Bits);
	
	/**
	 * Reads bus pins. The BUS_RD pin is set low, pins are set after a 50ns delay, and finally the BUS_RD pin is set high again.
	 */
	uint16_t bus_Read(void);
	
	/**
	 * Sets directions of bus pins.
	 * Lower 8 bits contain pin directions. 1 -> input, 0 -> output. Upper 8 bits are ignored.
	 */
	void bus_Set(uint16_t IOMap);
	
	/**
	 * Writes to bus. Bus pins must first be set to output. Lower 8 bits of argument are written to pins, and then the BUS_WR pin is brought low for approximately 50ns.
	 */
	void bus_Write(uint16_t Bits);
	
	/**
	 * Sets a pin high. If the pin is not already an output, it is made an output.
	 * @return 1 if the pin number was legal.
	 */
	uint16_t pin_HI(Picaso::Pin Pin);
	
	/**
	 * Sets a pin low. If the pin is not already an output, it is made an output.
	 * @return 1 if the pin number was legal.
	 */
	uint16_t pin_LO(Picaso::Pin Pin);
	
	/**
	 * Reads a pin.
	 */
	uint16_t pin_Read(Picaso::Pin Pin);
	
	/**
	 * Sets a pin as an input or an output.
	 * @param Mode 1 or Picaso::INPUT for input, 0 or Picaso::OUTPUT for output.
	 * @param Pin Pin to be set.
	 * @return 1 if pin number is legal.
	 */
	uint16_t pin_Set(uint16_t Mode, Picaso::Pin Pin);
	
	
	
	// Serial communication command:
	
	/**
	 * Set baud rate of serial interface with display.
	 */
	void setbaudWait(Picaso::BaudRate Newrate);
	
	

	private:
	
	#if LCD_USING_MODSERIAL
	MODSERIAL serial;
	#else
	Serial serial;
	#endif
	
	DigitalOut rst;
	Timer timer;
	
	void WriteBytes(char* psOutput, int count);
	void WriteChars(char* psOutput);
	void WriteWords(uint16_t* source, int count);
	int ReadSerPort(char *psData, int iMax);
	void getbytes(char *data, int size);
	void GetAck();
	uint16_t GetWord();
	void getString(char *outStr, int strLen);
	uint16_t GetAckResp();
	uint16_t WaitForAck();
	uint16_t GetAckRes2Words(uint16_t * word1, uint16_t * word2);
	void GetAck2Words(uint16_t * word1, uint16_t * word2);
	uint16_t GetAckResSector(char* Sector);
	uint16_t GetAckResStr(char * OutStr);
	uint16_t GetAckResData(char* OutData, uint16_t size);
	void SetThisBaudrate(Picaso::BaudRate NewRate);
	
	void LCD_wait_ms(int ms);
	void rxFlush();
	
	int Error4D ;  			// Error indicator,  used and set by Intrinsic routines
	int TimeLimit4D;		// time limit in ms for total serial command duration, 2000 (2 seconds) should be adequate for most commands
	                        // assuming a reasonable baud rate AND low latency AND 0 for the Serial Delay Parameter
	                        // temporary increase might be required for very long (bitmap write, large image file opens)
	                        // or indeterminate (eg file_exec, file_run, file_callFunction)  commands
};
#endif