#ifndef PICASO_SERIAL_H
#define PICASO_SERIAL_H

#include "mbed.h"
#include <string>

/**
* Library for 3.2'' uLcd Picaso Display
* 4D system
* Picaso Serial Environment Command Set
* http://www.4dsystems.com.au/product/20/67/Processors_Graphics/PICASO/
* @autor A.Decarvalho
*
*17/02/ 2015
*/

//--------------------
// display Gfx
//--------------------
#define CMD_TOUCH_SET   0xFF38
#define CMD_TOUCH_GET   0xFF37
#define CMD_TOUCH_DETECT_REGION 0xFF39

#define CMD_BAUD_RATE   0x0026
#define CMD_CLEAR_SCREEN    0xFFCD
#define CMD_BACKGROUND_COLOR    0xFFA4

#define CMD_FILLED_RECT 0xFFC4
#define CMD_RECT    0xFFC5

#define CMD_CIRCLE  0xFFC3
#define CMD_FILLED_CIRCLE   0xFFC2

#define CMD_LINE    0xFFC8

#define CMD_TRIANGLE    0xFFBF
#define CMD_FILLED_TRIANGLE 0xFFA9

#define CMD_PIXEL   0xFFC1

#define CMD_PANEL   0xFFAF

#define CMD_BUTTON  0x0011
#define BUTTON_RAISED   0x0001
#define BUTTON_DEPRESSED    0x0000

#define CMD_SLIDER   0xFFAE
#define SLIDER_INTENTED 0x0000
#define SLIDER_RAISED   0x0001
#define SLIDER_HIDDEN   0x0002

#define CMD_PUT_STRING  0x0018
#define CMD_MOVE_ORIGIN 0xFFCC
#define CMD_TEXT_FOREGROUND 0xFFE7
#define CMD_TEXT_BACKROUND 0xFFE6
#define CMD_TEXT_SET_FONT   0xFFE5

#define CMD_POLYGON     0x0013
#define CMD_FILLED_POLYGON  0x0014
#define CMD_ELLIPSE         0xFFB2
#define CMD_FILLED_ELLIPSE  0xFFB1

#define GFX_ACK 0x06
#define GFX_NAK 0x00

//speed baud
#define GFX_BAUD_9600      0x0006
#define GFX_BAUD_19200     0x0008
#define GFX_BAUD_56000     11
#define GFX_BAUD_115200    0x000D
#define GFX_BAUD_128000  0x000E
#define GFX_BAUD_256000  0x000F

//-----------------
typedef char* XCHAR;
typedef unsigned short UINT16;
typedef unsigned char UINT8;
typedef short INT16;

//--------------------
// The States
//--------------------
enum State {
    BUTTON_STATE_PRESSED = 0,
    BUTTON_STATE_RELEASED=1,
    //
    LED_STATE_ON,
    LED_STATE_OFF,
    //
    SWITCHER_STATE_ON,
    SWITCHER_STATE_OFF,
    //
    SLIDER_STATE_MOVE,
    SLIDER_STATE_RELEASED,
    //
    SCALER_STATE_PRESSED_UP,
    SCALER_STATE_PRESSED_DOWN,
    SCALER_STATE_RELEASED,
    //
    LABEL_STATE_ON,
    //
    DIGIT_LED_STATE_ON,
    //
    STATE_ON,
    STATE_OFF,
    BUMP_LEFT,
    BUMP_RIGHT,
    BUMP_DOWN,
    BUMP_UP,
    STATE_ROTATE
};
//------------
// fonts uLCD
//------------
enum Font {
    FONT_1 = 0x0000,
    FONT_2 = 0x0001,
    FONT_3 = 0x0002
};
//-----------------
// Colors for Obj
//-----------------
enum Color {
    ALICEBLUE = 0xF7DF,ANTIQUEWHITE = 0xFF5A,AQUA = 0x07FF,
    AQUAMARINE = 0x7FFA,AZURE = 0xF7FF,BEIGE = 0xF7BB,
    BISQUE = 0xFF38,BLACK = 0x0000,BLANCHEDALMOND = 0xFF59,
    BLUE = 0x001F,BLUEVIOLET = 0x895C,BROWN = 0xA145,
    BURLYWOOD = 0xDDD0,CADETBLUE = 0x5CF4,CHARTREUSE = 0x7FE0,
    CHOCOLATE = 0xD343,CORAL = 0xFBEA,CORNFLOWERBLUE = 0x64BD,
    CORNSILK = 0xFFDB,CRIMSON = 0xD8A7,CYAN = 0x07FF,
    DARKBLUE = 0x0011,DARKCYAN = 0x0451,DARKGOLDENROD = 0xBC21,
    DARKGRAY = 0xAD55,DARKGREEN = 0x0320,DARKKHAKI = 0xBDAD,
    DARKMAGENTA = 0x8811,DARKOLIVEGREEN = 0x5345,DARKORANGE = 0xFC60,
    DARKORCHID = 0x9999,DARKRED = 0x8800,DARKSALMON = 0xECAF,
    DARKSEAGREEN = 0x8DF1,DARKSLATEBLUE = 0x49F1,DARKSLATEGRAY = 0x2A69,
    DARKTURQUOISE = 0x067A,DARKVIOLET = 0x901A,DEEPPINK = 0xF8B2,
    DEEPSKYBLUE = 0x05FF,DIMGRAY = 0x6B4D,DODGERBLUE = 0x1C9F,
    FIREBRICK = 0xB104,FLORALWHITE = 0xFFDE,FORESTGREEN = 0x2444,
    FUCHSIA = 0xF81F,GAINSBORO = 0xDEFB,GHOSTWHITE = 0xFFDF,
    GOLD = 0xFEA0,GOLDENROD = 0xDD24,GRAY = 0x8410,
    GREEN = 0x0400,GREENYELLOW = 0xAFE5,HONEYDEW = 0xF7FE,
    HOTPINK = 0xFB56,INDIANRED = 0xCAEB,INDIGO = 0x4810,
    IVORY = 0xFFFE,KHAKI = 0xF731,LAVENDER = 0xE73F,
    LAVENDERBLUSH = 0xFF9E,LAWNGREEN = 0x7FE0,LEMONCHIFFON = 0xFFD9,
    LIGHTBLUE = 0xAEDC,LIGHTCORAL = 0xF410,LIGHTCYAN = 0xE7FF,
    LIGHTGOLD = 0xFFDA,LIGHTGREEN = 0x9772,LIGHTGREY = 0xD69A,
    LIGHTPINK = 0xFDB8,LIGHTSALMON = 0xFD0F,LIGHTSEAGREEN = 0x2595,
    LIGHTSKYBLUE = 0x867F,LIGHTSLATEGRAY = 0x7453,LIGHTSTEELBLUE = 0xB63B,
    LIGHTYELLOW = 0xFFFC,LIME = 0x07E0,LIMEGREEN = 0x3666,
    LINEN = 0xFF9C,MAGENTA = 0xF81F,MAROON = 0x8000,
    MEDIUMAQUAMARINE = 0x6675,MEDIUMBLUE = 0x0019,MEDIUMORCHID = 0xBABA,
    MEDIUMPURPLE = 0x939B,MEDIUMSEAGREEN = 0x3D8E,MEDIUMSLATEBLUE = 0x7B5D,
    MEDIUMSPRINGGREEN = 0x07D3,MEDIUMTURQUOISE = 0x4E99,MEDIUMVIOLETRED = 0xC0B0,
    MIDNIGHTBLUE = 0x18CE,MINTCREAM = 0xF7FF,MISTYROSE = 0xFF3C,
    MOCCASIN = 0xFF36,NAVAJOWHITE = 0xFEF5,NAVY = 0x0010,
    OLDLACE = 0xFFBC,OLIVE = 0x8400,OLIVEDRAB = 0x6C64,
    ORANGE = 0xFD20,ORANGERED = 0xFA20,ORCHID = 0xDB9A,
    PALEGOLDENROD = 0xEF55,PALEGREEN = 0x9FD3,PALETURQUOISE = 0xAF7D,
    PALEVIOLETRED = 0xDB92,PAPAYAWHIP = 0xFF7A,PEACHPUFF = 0xFED7,
    PERU = 0xCC27,PINK = 0xFE19,PLUM = 0xDD1B,
    POWDERBLUE = 0xB71C,PURPLE = 0x8010,RED = 0xF800,
    ROSYBROWN = 0xBC71,ROYALBLUE = 0x435C,SADDLEBROWN = 0x8A22,
    SALMON = 0xFC0E,SANDYBROWN = 0xF52C,SEAGREEN = 0x2C4A,
    SEASHELL = 0xFFBD,SIENNA = 0xA285,SILVER = 0xC618,
    SKYBLUE = 0x867D,SLATEBLUE = 0x6AD9,SLATEGRAY = 0x7412,
    SNOW = 0xFFDF,SPRINGGREEN = 0x07EF,STEELBLUE = 0x4416,
    TAN = 0xD5B1,TEAL = 0x0410,THISTLE = 0xDDFB,TOMATO = 0xFB08,
    TURQUOISE = 0x471A,VIOLET = 0xEC1D,WHEAT = 0xF6F6,
    WHITE = 0xFFFF,WHITESMOKE = 0xF7BE,YELLOW = 0xFFE0,
    YELLOWGREEN = 0x9E66
};
//------------------------------------------------------
enum TouchEvent {
    EVENT_NO_ACTIVITY = 0x0000,
    EVENT_PRESS = 0x0001,
    EVENT_RELEASE = 0x0002,
    EVENT_MOVE = 0x0003
};
//-------------------------------------------------------
enum WIDGET_TYPE {
    OBJ_BUTTON=1,
    OBJ_LED,
    OBJ_LABEL,
    OBJ_SCALER,
    OBJ_SLIDER,
    OBJ_SWITCHER,
    OBJ_DIGIT_LED,
    OBJ_DIGITAL_3_LED,

    OBJ_RECTANGLE,
    OBJ_CIRCLE,
    OBJ_TRIANGLE,
    OBJ_ELLIPSE,
    OBJ_DIAMOND,
    OBJ_ISOCELE,
    OBJ_SEGMENT,
    OBJ_UNKNOW
};
//-------------------------------------------------------
struct GraphicMessage {
    TouchEvent event;
    UINT16 objId;
    int posiX;
    int posiY;
};
//--------------------------------------------------------
class PicasoSerial:public Serial
{
public:
    /** Construct a PicasoSerial object.
    *
    * @param _pinTx the Serial Mbed pin Tx
    * @param _pinRx the Serial Mbed pin Rx
    * @param _pinReset the LCD reset pin
    */
    PicasoSerial(PinName _pinTx,PinName _pinRx,PinName _pinReset);

//------------------------------
    /**    @returns the touch status
    *      EVENT_NO_ACTIVITY,
    *      EVENT_PRESS,
    *      EVENT_RELEASE,
    *      EVENT_MOVE
    */
    TouchEvent touch_get_status();
//------------------------------
    //! @returns the touch X position
    UINT16 touch_get_x();
//------------------------------
    //! @returns the touch Y position
    UINT16 touch_get_y();
//-----------------------------
    //! reset the screen
    void gfx_reset();
//-----------------------------
    //! The first function is called, initialization of the display with the transmission baud
    void setup(int baud);
//-----------------------------
//-----------------------------
    //! @param speed the baud rate
    void gfx_set_UartSpeed(int baud);
//-----------------------------
    //! clear the screen
    void gfx_clearScreen();
//-----------------------------
    //!@params color define the background color with RGB format
    void gfx_set_BackGroundColor(Color color);
//---------------------------------------------
    //! create a color
    //! @params red
    //! @param green
    //! @param blue
    unsigned short gfx_set_RGB565(unsigned char red,
                                  unsigned char green, unsigned char blue);
//------------------------------------------------------------------------
    /** Draw a circle
    * @params x center position
    * @params y center position
    * @params radius
    * @params color
    */
    void draw_circle(UINT16 x, UINT16 y, UINT16 radius,Color color);
//------------------------------------------------------------------
    /** Draw a filled circle
    * @params x center position
    * @params y center position
    * @params radius
    * @params color
    */
    void draw_filled_circle(UINT16 x, UINT16 y, UINT16 radius,Color color);
//---------------------------------------------------------------------------
    /** Draw a Line [x1,y1] to [x2,y2]
    * @params x1
    * @params y1
    * @params x2
    * @params y2
    * @params color
    */
    void draw_line(UINT16 x1, UINT16 y1, UINT16 x2, UINT16 y2,Color color);
//---------------------------------------------------------------------------
    //! Draw a triangle  [x1,y1],[x2,y2],[x3,y3]

    void draw_triangle(UINT16 x1, UINT16 y1, UINT16 x2,
                       UINT16 y2, UINT16 x3, UINT16 y3,Color color);
//---------------------------------------------------------------------------
    //! Draw a filled triangle  [x1,y1],[x2,y2],[x3,y3]

    void draw_filled_triangle(UINT16 x1, UINT16 y1, UINT16 x2,
                              UINT16 y2, UINT16 x3, UINT16 y3,Color color);
//---------------------------------------------------------------------------
    /** Draw a filled rectangle
    * @params x1 up left
    * @params y1 up left
    * @params x2 low right
    * @params y2 low right
    * @params color
    */

    void draw_filled_rectangle(UINT16 x1, UINT16 y1, UINT16 x2,
                               UINT16 y2,Color color);
//---------------------------------------------------------------
    /** Draw a rectangle
       * @params x1 up left
       * @params y1 up left
       * @params x2 low right
       * @params y2 low right
       * @params color
       */
    void draw_rectangle(UINT16 x1, UINT16 y1, UINT16 x2,
                        UINT16 y2,Color color);
//------------------------------------------------------------------
    /** Draw a panel
       * @params state
       * @params x1 up left
       * @params y1 up left
       * @params x2 low right
       * @params y2 low right
       * @params color
       */
    void draw_panel(UINT16 state, UINT16 x1, UINT16 y1, UINT16 width,
                    UINT16 height,Color color);
//-------------------------------------------------------------------
    /** Draw a color pixel
       * @params x1
       * @params y1
       * @params color
       */
    void draw_pixel(UINT16 x1, UINT16 y1,Color color);
//--------------------------------------------------------------------
    /** Draw a button
       * @params *s the text button
       * @params x1 up left
       * @params y1 up left
       * @params font
       * @params etat BUTTON_RAISED or BUTTON_DEPRESSED
       * @params button_color the Bkg color
       * @params text_color
       */
    void draw_button(string str, UINT16 x1, UINT16 y1,
                     Font font,State state,Color button_color,Color text_color);
//-------------------------------------------------------------------------------------
    /** Draw a slider
    * @params mode
    * @params x1 up left
    * @params y1 up left
    * @params x2 low right
    * @params y2 low right
    * @params color the bkg color
    * @params scale the max value
    * @params value
    */
    void draw_slider(UINT16 mode, UINT16 x1, UINT16 y1, UINT16 x2, UINT16 y2,
                     Color color, UINT16 scale, UINT16 value);
//-----------------------------------------------------------------------------
    void text_foreground(Color color);
//-------------------------------
    void text_backround(Color color);
//------------------------------
    void text_putstring(string str);

//-------------------------------
    void text_setFont(Font _font);
//---------------------------------
    void gfx_move_to(UINT16 x1, UINT16 y1);
//-----------------------------------------
    /** Draw a string
        * @params *s the text
        * @params x up left
        * @params y up left
        * @params _font
        * @params text_color
        * @params color_bckg the background color
        */
    void draw_string(string str, UINT16 x, UINT16 y,Font _font,Color text_color,Color color_bckg);

//---------------------------------------------------------------------------------
    //! Draw a diamond [x1,y1],[x2,y2],[x3,y3],[x4,y4]

    void draw_diamond(UINT16 x1, UINT16 y1, UINT16 x2,UINT16 y2,UINT16 x3,UINT16 y3,
                      UINT16 x4,UINT16 y4,Color color);
//----------------------------------------------------------------------------------------
    //! Draw a filled diamond [x1,y1],[x2,y2],[x3,y3],[x4,y4]
    void draw_filled_diamond(UINT16 x1, UINT16 y1, UINT16 x2,UINT16 y2,UINT16 x3,UINT16 y3,
                             UINT16 x4,UINT16 y4,Color color);
//-------------------------------------------------------------
    //! Draw a ellipse
    void draw_ellipse(UINT16 xc, UINT16 yc, UINT16 xrad,UINT16 yrad,Color color);
//--------------------------------------------------------------------------------
    //! Draw a filled ellipse
    void draw_filled_ellipse(UINT16 xc, UINT16 yc, UINT16 xrad,UINT16 yrad,Color color);

protected:

    DigitalOut  myPinReset;


    void touch_set(UINT16 mode);
    void touch_detect_region(UINT16 x1, UINT16 y1, UINT16 x2, UINT16 y2);
    void put_gfx_char(unsigned char data);
    void wait_gfx_ack();
    UINT8 get_gfx_char(void);
    void put_gfx_Commande(unsigned char* commande, int nombre);

};

#endif

