#ifndef IP12B512_H
#define IP12B512_H

#include "mbed.h"

// Operation Instruction Set
#define IP12B512_READ  0x03 // Read memory data beginning at selected address
#define IP12B512_WRITE 0x02 // Write memory data beginning at selected address
#define IP12B512_RDSR  0x05 // Read status register (not implemented)
#define IP12B512_WRSR  0x01 // Write status register
#define IP12B512_RDMI  0x0E // Read Memory Size

/** IP12B512 class.
 *  Read/write byte and read/write chunks from/to IP12B512 SPI RAM
 * 
 *  PDF manual @ http://www.search-pdf-manuals.com/manual/ipsilog-ip12b512-512k-spi-low-power-serial-sram-46
 * 
 * Notes:
 * - the MOSI and MISO pins must have pulldown resistors (i used 5k6 resistors)
 * - the SCLK pin has a pulldown resistor (i used 5k6 resistor)
 * - the HOLD pin is connected directly to VCC
 * - the CS pin has a pullup resistor (i used 5k6 resistor)
 *
 * Example:
 * @code
 *
 * #include "mbed.h"
 * #include "IP12B512.cpp"
 *
 * IP12B512 sram(SPI_MOSI, SPI_MISO, SPI_SCK, D6); // MOSI, MISO, SCK, CS
 *
 * int main() {
 *     // GET SRAM SIZE
 *     uint32_t ram_size = sram.GetRamSize();
 *     printf("ram_size %d\n", ram_size);
 *     
 *     // CLEAR ALL SRAM
 *     sram.ClearAll();
 *     
 *     // WRITE A SINGLE BYTE TO SRAM
 *     uint16_t ram_addr = 0;
 *     uint8_t sent_data = 0x6F;
 *     sram.Write(ram_addr, sent_data);
 *     printf("written at address %d data 0x%02X\n", ram_addr, sent_data);
 *     
 *     // READ A SINGLE BYTE FROM RAM
 *     uint8_t rec_data = sram.Read(ram_addr);
 *     printf("read at address %d data 0x%02X\n", ram_addr, rec_data);
 * }
 * @endcode
 */
class IP12B512
{
    public:
/** Constructor: setup the SPI and write to "Status Register" that we are going to use it in "Virtual Chip Mode"
 * 
 * @param pin_mosi the hardware MOSI pin on the board
 * @param pin_miso the hardware MISO pin on the board
 * @param pin_sclk the hardware SCLK pin on the board
 * @param pin_cs the software CS pin on the board (can be any output pin)
 * 
 */
        IP12B512(
            PinName pin_mosi, 
            PinName pin_miso, 
            PinName pin_sclk, 
            PinName pin_cs
        );
/** Write SRAM in byte mode (sends the highest amount of data to SRAM prior to write)
 */
        void Write(uint16_t addr, uint8_t data);
/** Write SRAM in stream mode (faster then Write method)
 */
        void StreamWrite(uint16_t addr, uint8_t *data, uint32_t size);
/** Read SRAM in byte mode (sends the highest amount of data to SRAM prior to read)
 */
        uint8_t Read(uint16_t addr);
/** Read SRAM in stream mode (faster then Read method)
 */
        void StreamRead(uint16_t addr, uint8_t *data, uint32_t size);
/** Fill SRAM with data
 */
        void ClearAll();
/** Get the SRAM size in bytes
 */
        uint32_t GetRamSize();

    private:
        SPI _device;
        DigitalOut _cs;
};

#endif