// This is the code for the RENBED microcontroller board which is connected to the LMAB Measurement Board V1.0
// RENISHAW d.o.o.
// ab138110; adam.balkan@renishaw.com
// August 2016

#include "mbed.h"
#include "USBSerial.h"

USBSerial pc;                           // For talking to the PC
SPI dac_adc_link(P1_22, P1_21, P1_20);  // MOSI, MISO, SCLK  (for talking to the DAC)
DigitalOut DAC_CS(P1_26);               // CS (chip select) for selecting the DAC        
DigitalInOut OEb(P1_14);                // Output enable active low
DigitalInOut LE(P0_17);                 // Latch enable active low
DigitalInOut PDb(P0_7);                 // Power down active low

char command = 0x00;                    // stores the command that was sent from the PC
char param_p1 = 0x00;                   // stores the first "parameter" byte from the PC
char param_p2 = 0x00;                   // stores the second "parameter" byte from the PC
char param_p3 = 0x00;                   // stores the third "parameter" byte from the PC
char param_p4 = 0x00;                   // stores the fourth "parameter" byte from the PC
    
void Disconnect_PDb()
{
  PDb.input();              // by connecting PDb as an input, we effectively make it "open"
  PDb.mode(PullNone);       // we need to make sure that there is no pull-up or pull down resistor on PDb when it is open
}

void Disconnect_OEb()
{
   OEb.input();             // by connecting OEb as an input, we effectively make it "open"
   OEb.mode(PullNone);      // we need to make sure that there is no pull-up or pull down resistor on OEb when it is open    
}

void Disconnect_LE()
{
  LE.input();               // by connecting LE as an input, we effectively make it "open"
  LE.mode(PullNone);        // we need to make sure that there is no pull-up or pull-down resistor on LE when it is open
}

void Connect_PDb()
{
  PDb.output();             // to re-connect PDb we configure it as an output
}

void Connect_OEb()
{
   OEb.output();            // to re-connect OEb we configure it as an output
}

void Connect_LE()
{
  LE.output();              // to re-connect LE we configure it as an output
}


void send_analog_data_to_pc(char id, char data1, char data2)
{
        pc.putc(id);      // ID of analog output
        pc.putc(data1);   // Part 1 of analog output       
        pc.putc(data2);   // Part 2 of analog output
        pc.putc(13);      // Carriage return
        wait_ms(100);
}


void SET_ASIC_VDD(int desired_value) // sets ASIC VDD using DAC MCP4921
{
      
        char command_register_msb = (0x70);                                 // 01110000 for DAC A, BUF, Gain =1  and no shutdown
        command_register_msb = ((command_register_msb)|desired_value>>8);   // assign first 4 bits to first byte to send
        char command_register_lsb = desired_value;                          // assign last 8 bits to second byte to send
        DAC_CS =0;                                                          // bring CS low
        int junk= dac_adc_link.write(command_register_msb);                 // write first byte
        junk= dac_adc_link.write(command_register_lsb);                     // write second byte
        DAC_CS=1;                                                           // De-select device 
        wait_ms(100);                                                       // wait for 100 ms
}


int main() {
     dac_adc_link.format(8,0);                  // Specify that the SPI link to the DAC should be 8 bits long and be in mode 0     
     Connect_PDb();                             // On start-up, connect PDb
     Connect_LE();                              // On start-up, connect LE
     Connect_OEb();                             // On start-up, connect OEb
     PDb = 1;                                   // Power down is inactive on start up
     LE = 1;                                    // Latch enable is disabled on start up
     OEb = 0;                                   // Output enable is enabled on start up
     DAC_CS = 1;                                // DAC is deselected on start-up     
        
    while(1) {

        if (pc.readable()){             // If the PC has sent a command to the Renbed...

        command = pc.getc();    // Data from PC (command) (the first byte)
        param_p1 = pc.getc();   // Data from PC (param p1) (the second byte)
        param_p2 = pc.getc();   // Data from PC (param p2)  (the third byte)
        param_p3 = pc.getc();   // Data from PC (param p2) (the fourth byte)
        param_p4 = pc.getc();   // Data from PC (param p2) (the fifth byte)
        int scrap = pc.getc();    // Carriage Return (the sixth byte) (we scrap this byte)
  
        if (command=='E') // request to change ASIC VDD
        {
        int desired_value = 256*param_p3+param_p4; // represents desired ASIC VDD as an integer between 0 (0V) and 4096 (5V) for example 2703 is 3.3 volts
        SET_ASIC_VDD(desired_value);
        send_analog_data_to_pc('E', 0x00,  0x00); // lower part followed by upper part
        }
        else if (command=='Z') // toggle PDb
        {
        PDb = !PDb;
        send_analog_data_to_pc('Z',  0x00,  0x00); // lower part followed by upper part
        }
        else if (command=='X') // toggle latch enable
        {
        LE =!LE;
        send_analog_data_to_pc('X',  0x00,  0x00); // lower part followed by upper part
        }
        else if (command=='C') // toggle output enable
        {
        OEb = !OEb;
        send_analog_data_to_pc('C', 0x00, 0x00); // lower part followed by upper part
        }
        else if (command=='O')  // Disconnect PDb (make it open)
        {
        Disconnect_PDb();
        send_analog_data_to_pc('O', 0x00, 0x00); // lower part followed by upper part
        }
        else if (command=='F')  // Disconnect OEb (make it open)
        {
        Disconnect_OEb();
        send_analog_data_to_pc('F', 0x00, 0x00); // lower part followed by upper part
        }
        else if (command=='W')  // Disconnect LE (make it open)
        {
        Disconnect_LE();
        send_analog_data_to_pc('W', 0x00, 0x00); // lower part followed by upper part
        }
        else if (command=='o')  // Connect PDb 
        {
        Connect_PDb();
        send_analog_data_to_pc('o', 0x00, 0x00); // lower part followed by upper part
        }
        else if (command=='f')  // Connect OEb
        {
        Connect_OEb();
        send_analog_data_to_pc('f', 0x00, 0x00); // lower part followed by upper part
        }
        else if (command=='w')  // Connect LE
        {
        Connect_LE();
        send_analog_data_to_pc('w', 0x00, 0x00); // lower part followed by upper part
        }
        else
        {
        pc.printf("Invalid command\n");     // if an unrecognised command was sent from the PC
        }
      } 
    }
}
