#include "mbed.h"

InterruptIn sig(D5);
Timer sigTimer;
Serial pc(USBTX, USBRX);

int sigPeriod = 0;

// Define display ports
const PinName cellPins[] = {A0, A1, A2, A3};
const PinName segPins[] = {D13, D12, D11, D10, D9, D8, D7, D6};
                        //  A    B    C    D   E   F   G   Dot
DigitalOut *cellPorts[4];
DigitalOut *segPorts[8];

Timer displayTimer;

namespace SevenSegments{
    enum Element {
      SegA = 1,
      SegB = 2,
      SegC = 4,
      SegD = 8,
      SegE = 16,
      SegF = 32,
      SegG = 64,
      SegDot = 128
    };
    
    const uint8_t digits[] = {
      SegA | SegB | SegC | SegD | SegE | SegF,
      SegB | SegC,
      SegA | SegB | SegG | SegE | SegD,
      SegA | SegB | SegG | SegC | SegD,
      SegF | SegB | SegG | SegC,
      SegA | SegF | SegG | SegC | SegD,
      SegA | SegF | SegG | SegE | SegC | SegD,
      SegA | SegB | SegC,
      SegA | SegB | SegC | SegD | SegE | SegF | SegG,
      SegA | SegB | SegG | SegF | SegC | SegD
    };
};

uint8_t cellValues[4];

Timer periodUpdateTimer;

// Function prototypes
void displaySevenSegs(uint8_t cellValues[]);
void valueToBcd(int val, uint8_t cellValues[]);
void updateDisplayedValue(int val, uint8_t cellValues[]);
void onRisingEdge();

int main()
{
    displayTimer.start();
    // Initialize cell ports
    for (int i = 0; i < 4; i++) {
      cellPorts[i] = new DigitalOut(cellPins[i]);
      *cellPorts[i] = 1;
    }
    // Initialize segment ports
    for (int i = 0; i < 8; i++)
      segPorts[i] = new DigitalOut(segPins[i]);

    sig.mode(PullNone);
    sig.rise(onRisingEdge);
    sigTimer.start();
    periodUpdateTimer.start();
    while(1) {
        displaySevenSegs(cellValues);
        updateDisplayedValue(sigPeriod, cellValues);
    }
}

void onRisingEdge()
{
    static int queue[128], queueStart = 0, queueEnd = 0;
    static uint64_t sum = 0;
    int time;
    
    time = sigTimer.read_us();
    sigTimer.reset();
    if (queueStart == queueEnd && sum != 0) {
        sum -= queue[queueStart];
        queueStart = (queueStart + 1) & 127;
    }
    sum += time;
    queue[queueEnd] = time;
    queueEnd = (queueEnd + 1) & 127;
    
    if (queueStart == queueEnd)
        sigPeriod = sum / 128;
    else
        sigPeriod = sum / ((queueEnd + 128 - queueStart) & 127);
}

void bcdToSevenSegs(uint8_t val)
{
  uint8_t segs;

  segs = SevenSegments::digits[val];
  for (int s = 0; s < 8; s++) {
    if (segs & (1 << s))
      // Enable segment
      *segPorts[s] = 1;
    else
      // Disable segment
      *segPorts[s] = 0;
  }
}

void displaySevenSegs(uint8_t cellValues[])
{
    static bool activeCells[5] = {true, false, false, false, false};
        
    if (displayTimer.read_us() < 1000)
        return;
    displayTimer.reset();
    for (int i = 4; i > 0; i--)
    activeCells[i] = activeCells[i - 1];
    activeCells[0] = activeCells[4];
    // Turn off all cells
    for (int i = 0; i < 4; i++)
    *cellPorts[i] = 1;
    // Change cell value
    for (int i = 0; i < 4; i++)
    if (activeCells[i])
        bcdToSevenSegs(cellValues[i]);
    // Turn on active cell
    for (int i = 0; i < 4; i++)
    if (activeCells[i])
      *cellPorts[i] = 0;
}


void valueToBcd(int val, uint8_t cellValues[])
{
  for (int i = 3; i >= 0; i--) {
    cellValues[i] = val % 10;
    val /= 10;
  }
}

void updateDisplayedValue(int val, uint8_t cellValues[])
{
    if (periodUpdateTimer.read_ms() < 100)
        return;
    periodUpdateTimer.reset();
    valueToBcd(val, cellValues);
}