#include "mbed.h"

// Define display ports
const PinName cellPins[] = {A0, A1, A2, A3};
const PinName segPins[] = {D13, D12, D11, D10, D9, D8, D7, D6};
                        //  A    B    C    D   E   F   G   Dot
DigitalOut *cellPorts[4];
DigitalOut *segPorts[8];

Timer displayTimer;

namespace SevenSegments{
    enum Element {
      SegA = 1,
      SegB = 2,
      SegC = 4,
      SegD = 8,
      SegE = 16,
      SegF = 32,
      SegG = 64,
      SegDot = 128
    };
    
    const uint8_t digits[] = {
      SegA | SegB | SegC | SegD | SegE | SegF,
      SegB | SegC,
      SegA | SegB | SegG | SegE | SegD,
      SegA | SegB | SegG | SegC | SegD,
      SegF | SegB | SegG | SegC,
      SegA | SegF | SegG | SegC | SegD,
      SegA | SegF | SegG | SegE | SegC | SegD,
      SegA | SegB | SegC,
      SegA | SegB | SegC | SegD | SegE | SegF | SegG,
      SegA | SegB | SegG | SegF | SegC | SegD
    };
};


// Rotary encoder specific objects
DigitalIn A(D15);
DigitalIn B(D14);

Timer encoderTimer;

// PC serial connection
Serial pc(USBTX, USBRX);

int counter = 0;
uint8_t cellValues[4];

// Function prototypes
void readEncoder();
void encoderIncrement();
void encoderDecrement();
void displaySevenSegs(uint8_t cellValues[]);
void valueToBcd(int val, uint8_t cellValues[]);

int main()
{
    displayTimer.start();

    // Initialize cell ports
    for (int i = 0; i < 4; i++) {
      cellPorts[i] = new DigitalOut(cellPins[i]);
      *cellPorts[i] = 1;
    }
    // Initialize segment ports
    for (int i = 0; i < 8; i++)
      segPorts[i] = new DigitalOut(segPins[i]);
    
    // Change encoder pin mode
    A.mode(PullNone);
    B.mode(PullNone);

    while(1) {
        displaySevenSegs(cellValues);
        readEncoder();
        // fă altceva util
        // ...
        
    }
}

void bcdToSevenSegs(uint8_t val)
{
  uint8_t segs;

  segs = SevenSegments::digits[val];
  for (int s = 0; s < 8; s++) {
    if (segs & (1 << s))
      // Enable segment
      *segPorts[s] = 1;
    else
      // Disable segment
      *segPorts[s] = 0;
  }
}

void displaySevenSegs(uint8_t cellValues[])
{
    static bool activeCells[5] = {true, false, false, false, false};
        
    if (displayTimer.read_us() < 1000)
        return;
    displayTimer.reset();
    for (int i = 4; i > 0; i--)
    activeCells[i] = activeCells[i - 1];
    activeCells[0] = activeCells[4];
    // Turn off all cells
    for (int i = 0; i < 4; i++)
    *cellPorts[i] = 1;
    // Change cell value
    for (int i = 0; i < 4; i++)
    if (activeCells[i])
        bcdToSevenSegs(cellValues[i]);
    // Turn on active cell
    for (int i = 0; i < 4; i++)
    if (activeCells[i])
      *cellPorts[i] = 0;
}


void valueToBcd(int val, uint8_t cellValues[])
{
  for (int i = 3; i >= 0; i--) {
    cellValues[i] = val % 10;
    val /= 10;
  }
}

void readEncoder()
{
    static int state = 0;
    int currentState;
    const int stateOrder[] = {0, 2, 3, 1};
    const int stateIndex[] = {0, 3, 1, 2};
    int ccwState, cwState;
    
    currentState = (A << 1) + B;
    ccwState = stateOrder[(stateIndex[state] + 3) & 3];
    cwState = stateOrder[(stateIndex[state] + 1) & 3];
    if (currentState == cwState)
        encoderIncrement();
    else if (currentState == ccwState)
        encoderDecrement();
    state = currentState;
}

void encoderIncrement()
{
    counter++;
    valueToBcd(counter, cellValues);
    pc.printf("Counter is: %d.\n\r", counter);
}

void encoderDecrement()
{
    counter--;
    valueToBcd(counter, cellValues);
    pc.printf("Counter is: %d.\n\r", counter);    
}