/*
 *  Card / Proximity reader driver library
 *  Copyright (c) 2012 Neal Horman - http://www.wanlink.com
 *  
 *  License: MIT open source (http://opensource.org/licenses/MIT)
 *      Summary;
 *      Use / modify / distribute / publish it how you want and 
 *      if you use it, or don't, you can't hold me liable for how
 *      it does or doesn't work.
 *      If it doesn't work how you want, don't use it, or change
 *      it so that it does work.
 */
 
#ifndef _READER_H_
#define _READER_H_

#define KEYPAD_DIGIT_LIMIT 5
#define KEYPAD_TIMEOUT_MS 1500

class Reader
{
public:
    Reader()
        : mCard(0)
        , mFacility(0)
        , mSite(0)
        , mDigits(0)
        , mBits(0)
        , mBitsCount(0)
        , mbIsValid(false)
        , mbIsNew(false)
        , mbInhibited(false)
        {};
    
    uint32_t card() { return mCard; };
    uint32_t facility() { return mFacility; };
    bool isValid() { return mbIsValid; };
    bool isNew() { return mbIsNew; };
    void old() { mbIsNew = false; };
    
    bool isInhibited() { return mbInhibited; };
    void inhibit(bool bInhibit = true) { mbInhibited = bInhibit; };
    
    uint64_t bits() { return mLastBits; };
    uint8_t bitCount() { return mLastBitsCount; };

protected:
    volatile uint16_t mCard;
    volatile uint32_t mFacility;
    volatile uint16_t mSite;
    volatile uint8_t mDigits;
    volatile uint64_t mLastBits;
    volatile uint8_t mLastBitsCount;
    
    volatile uint64_t mBits;
    volatile uint8_t mBitsCount;
    
    volatile bool mbIsValid;
    volatile bool mbIsNew;
    
    bool mbInhibited;
    Timeout mKeypadTimer;
    
    void keypadTimerRestart()
    {
        mKeypadTimer.detach();
        mKeypadTimer.attach_us(callback(this, &Reader::keypadTimeout), KEYPAD_TIMEOUT_MS * 1000);
    }
    
    void keypadTimeout()
    {
        mDigits = 0;
        mbIsValid = true;
        codeSubmit();
    }

    void shiftIn(uint8_t bit)
    {
        mBits <<= 1;
        if (bit)
        {
            mBits |= (uint64_t)1;
        }
        mBitsCount++;
    }
    
    // param first: The index of the starting bit, counting from the most significant bit. (1 based)
    uint32_t bits32(uint8_t first, uint8_t qty)
    {   uint32_t val = 0;
        uint8_t start = mBitsCount - first;
        uint8_t end = start - qty;
    
        for(uint8_t i = start; i > end; i--)
        {
            val <<= 1;
            
            if (mBits & (1 << i))
            {
                val |= 1;
            }
        }
        
        return val;
    }
    
    bool decode()
    {
        bool bIsValid = false;
        
        if (mBitsCount == 4) // Key press
        {
            keypadTimerRestart();
            
            if (mDigits == 0) // Reset the card code for the first keypress
            {
                mCard = 0;
            }
            
            if (mBits < 10) // Number key
            {
                mCard *= 10;
                mCard += mBits;
                mDigits += 1;   
            }
            else if (mBits == 11 || mDigits == KEYPAD_DIGIT_LIMIT) // # Key (Enter)
            {
                bIsValid = true;
                mKeypadTimer.detach();
                mDigits = 0;
            }
        }
        else if (mBitsCount >= 26)
        {
            mCard = bits32(10, 16);
            bIsValid = true;
        }
        else
        {
            mFacility = 0;
            mCard = 0;
            mDigits = 0;
        }
        
        mLastBits = mBits;
        mLastBitsCount = mBitsCount;
        
        return bIsValid;
    }
    
    void swiped()
    {
        // decode mBits into mCard, mFacility, and mSite
        mbIsValid = !mbInhibited && decode();
        
        codeSubmit();
    }
    
    void codeSubmit()
    {
        // reset mBits and mBitsCount for next operation
        mBits = 0;
        mBitsCount = 0;
        mbIsNew = mbIsValid;
    }
};

#endif