/*
==========================================================================
Project:     Sump Pump Monitor
Description: Measure pump current to indicate run status. (10amp pump) 
Date:        11-6-16
Author:      Jim Abraham
Board:       WIZwiki-W7500
Rev:
==========================================================================
*/

#include "mbed.h"
#include "EthernetInterface.h"
#include <stdio.h>
#include <string.h>

#define USE_DHCP    0  // 0= no dhcp
#define MAC     "\x00\x08\xDC\x11\x34\x78"
#define IP      "192.168.0.12"
#define MASK    "255.255.255.0"
#define GATEWAY "192.168.0.254"
#define HTTPD_SERVER_PORT   80
#define HTTPD_MAX_REQ_LENGTH   1023
#define HTTPD_MAX_HDR_LENGTH   255
#define HTTPD_MAX_FNAME_LENGTH   127
#define HTTPD_MAX_DNAME_LENGTH   127

#if defined(TARGET_WIZwiki_W7500)
Serial uart(USBTX, USBRX);
#include "static_colors.h"
// LED R     : server listning status
// LED GREEN : socket connecting status Ok
// LED BLUE  : socket connecting status Busy
#endif

EthernetInterface eth;
TCPSocketServer server;
TCPSocketConnection client;

//========================================================
// Data Variables.
//========================================================
int LP1,LP2,LP3;
float AMP_Reading_on_AO, AMPreadingHI, AMPreading;
char buffer[HTTPD_MAX_REQ_LENGTH+1];
char httpHeader[HTTPD_MAX_HDR_LENGTH+1];

//-----------------------------------------------------------
// Initialize a pins to perform analog input
//-----------------------------------------------------------
AnalogIn   ain0(A0);

//-----------------------------------------------------------
void Serial_Interface_init(void)
//-----------------------------------------------------------
{
 // Serial Interface eth;
    // Serial port configuration: 
    // 9600 baud, 8-bit data, no parity, stop bit
    uart.baud(9600);
    uart.format(8, SerialBase::None, 1);
    uart.printf("Initializing\n");
    wait(1.0);   
}

//-----------------------------------------------------------
void ethernet_init(void)
//-----------------------------------------------------------
{
 //    EthernetInterface eth;
    uart.printf("Initializing Ethernet\n");
    #if USE_DHCP
    //eth.init Use DHCP
    // Use DHCP for WIZnetInterface
    int ret = eth.init((uint8_t*)MAC);    
    uart.printf("Connecting DHCP\n");
    #else
    // IP,mask,Gateway
    int ret = eth.init((uint8_t*)MAC,IP,MASK,GATEWAY);  
    uart.printf("Connecting (IP,mask,Gateway)\n");
    #endif
    wait(1.0);
    // Check Ethernet Link-Done
    uart.printf("Check Ethernet Link\r\n");
    
    if(eth.link() == true) 
    { 
     uart.printf("- Ethernet PHY Link - Done\r\n");
     COLOR(_RED_);
    }
    else 
    {
        uart.printf("- Ethernet PHY Link - Fail\r\n");
        COLOR(_BLACK_);
    }
    wait(1.0);
    if(!ret) 
    {
        uart.printf("Initialized, MAC: %s\r\n", eth.getMACAddress());
        ret = eth.connect();
     
     if(!ret) 
     {
        uart.printf("IP: %s, MASK: %s, GW: %s\r\n",
        eth.getIPAddress(), eth.getNetworkMask(), eth.getGateway());
        COLOR(_CYAN_);
     } 
     else 
     {
        uart.printf("Error ethernet.connect() - ret = %d\r\n", ret);
        COLOR(_YELLOW_);
        exit(0);
     }
    } 
    else 
    {
        uart.printf("Error ethernet.init() - ret = %d\r\n", ret);
     COLOR(_BLACK_);
     exit(0);
    }  
      
    wait(1.0);
 // TCPSocketServer server;
    server.bind(HTTPD_SERVER_PORT);
    server.listen();
    uart.printf("Server Listening\n");   
}

//-----------------------------------------------------------
void VAC_Zero_Cross(void)
//-----------------------------------------------------------
{
  LP1=0;                 // Indicator to exit while loop
  if (LP2==1)
    {   
//Waits for the waveform to be close to 'zero' (500 adc)
      while(LP1==0)
      {
        wait_us(100);
        AMPreading=ain0.read();

//check its within range. (+/- 5)
        if ((AMPreading < 0.5) && (AMPreading > 0.4)) 
        {
           LP1=1;
        }
      }
    }
  LP2=1;
}

//-----------------------------------------------------------
 void Measure_AMPS(void)
//-----------------------------------------------------------
// Measure AC Current.
// 12bit ADC, 10MHz = 0.1usec conversion time.
// 12bit ADC (0-3.3v = 0-4095 values) 3.3/4096 = 0.81mv
// 1.65v = 2047 = 0 amps.
//-----------------------------------------------------------
{
    AMPreadingHI=0.0;
    for (LP3=0; LP3<100; LP3++)
    {
       wait_us(200);
      AMPreading=ain0.read();
      if (AMPreadingHI < AMPreading) AMPreadingHI  =  AMPreading;
    }
    
    if (AMPreadingHI <0.51)
    {
       COLOR(_BLUE_);
    }
    else
    {
       COLOR(_GREEN_);
    }
    
    AMP_Reading_on_AO = AMPreadingHI;
    uart.printf("$%2.2f!", AMPreading);
    sprintf(httpHeader,"$%2.2f!", AMP_Reading_on_AO); 
    client.send(httpHeader,strlen(httpHeader));       //send amps to PC.
}


//===================================================================
int main(void)
//===================================================================
{
//RGB LED: 
//WHITE = program running.
//RED   = not connected to PC.
//GRN   = pump running.
//BLUE  = pump not running.

    Serial_Interface_init(); 
    ethernet_init();
    LP1=0;                 // Indicator to exit while loop
    LP2=0;
//-----------------------------------------------------------   
    while(true) 
//-----------------------------------------------------------
    {
        uart.printf("\nWait for new connection...\r\n");
        server.accept(client);
        client.set_blocking(false, 1500); // Timeout after (1.5)s
        uart.printf("Connection from: %s\r\n", client.get_address());
        
//-----------------------------------------------------------
        while(true) 
//-----------------------------------------------------------
        {            
            if(!client.is_connected())
             {
               COLOR(_RED_);
               break;        //exit while
             }
                // Mesure ADC 0 - Check Sump Pump Run Status
                VAC_Zero_Cross();
                Measure_AMPS();              
                wait(1.0);
                COLOR(_WHITE_);
        }
//-----------------------------------------------------------
        client.close();  //close connection to pc app.
        ethernet_init();
    }
}