/*!
 * PwmReader.h
 *
 * Read signal occupacy on a pin in a non blocking way using timer and interrupt
 *
 * Copyright (c) 2017 -  Alexandre Bouillot github.com/abouillot
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documnetation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to  whom the Software is
 * furished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS OR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mbed.h"

/*!
 * Class PwmReader lmeasure the time spend in high and low state on a digital pin
 * Uses useconds timer and IRQ
 */
class PwmReader
{
public:
    /** create a PwmReader object
     *
     *  @param pin Pin numer to be observed
     */
    PwmReader(PinName pin) : _pin(pin) {
        init();
    };
    /** Start the measurment
     *
     */
    void start();
    /** Stop the measurment
     *
     */
    void stop();
    /** return the time spend in low state
     *
     * @return the percentage of time spent in low state. value between 0.0 and 1.0
     */
    float occupacyLow() {
        return 100.0 * _down / (_down + _high);
    };
    /** return the time spend in high state
     *
     * @return the percentage of time spent in high state. value between 0.0 and 1.0
     */
    float occupacyHigh() {
        return 100.0 * _high / (_down + _high);
    };
    /** return the time spend in low state
      *
     * @return the percentage of time spent in low state. value in useconds
     */
    long down_us() {
        return _down;
    };
    /** return the time spend in high state
     *
     * @return the percentage of time spent in high state. value in useconds
     */
    long high_us() {
        return _high;
    };
    /** return the time spend in low state
    * @param round set round to tue to round the value (closest integer) - false by default
    * @return the time spent in low state. value in mseconds
    */
    long down_ms(bool round = false) {
        return (_down + (round ? 500 : 0)) / 1000;
    };
    /** return the time spend in high state
     * @param round set round to tue to round the value (closest integer) - false by default
     * @return the time spent in high state. value in mseconds
     */
    long high_ms(bool round = false) {
        return (_high + (round ? 500 : 0)) / 1000;
    };
    /** return the time spend in low state
     * @param round set round to tue to round the value (closest integer) - false by default
     * @return the time spent in low state. value in seconds
     */
    long down(bool round = false) {
        return (_down + (round ? 500000 : 0)) / 1000000;
    };
    /** return the time spend in high state
     * @param round set round to tue to round the value (closest integer) - false by default
     * @return the time spent in high state. value in seconds
     */
    long high(bool round = false) {
        return (_high + (round ? 500000 : 0)) / 1000000;
    };
private:
    void init();
    void pressedInt();
    void releasedInt();
    InterruptIn _pin;
    long _high;
    long _down;
    Timer _timer;
    int _start_state;
    long _last_toggle;
};
