/** Apeiros Robot class.
 *  Used for developing programs for the Apeiros Robot by Abe Howell's Robotics.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "Apeiros.h"
 *
 * Apeiros apeiros(SERIAL_TX, SERIAL_RX, 100, 100);
 * 
 * int main() {
 *     apeiros.Begin();
 *      while(1)
 *     {
 *        if (apeiros.IsSerialDataAvailable()) apeiros.ParseUartData();
 *     }
 * }
 * @endcode
 */

#ifndef MBED_APEIROS_H
#define MBED_APEIROS_H
 
#include "mbed.h"
 
#define Dwh 2.621       // wheel diameter
#define PI 3.14159      // PI
#define Cwh (Dwh * PI)  // wheel circumference
#define TSCALE 10       // convert to 10ths of an inch
#define INVTICK 4883    // this is 1 / 204.8 us, to avoid using floating point math
#define NCLKS 128       // number of encoder clocks per wheel rotation
#define Ktps ((Cwh * TSCALE * INVTICK) / NCLKS)
#define MaxMotorSpeed 255

// Hitec HS-81 accepts a 600 to 2400us pulse control signal. //
// Do not use a value that is outside of the HS-81 specification! //
#define MIN_GRIPPER_PULSE 1000
#define MAX_GRIPPER_PULSE 2200

#define MAX_BUZZER_PWM 100

/** Create Apeiros class
     *
     * @param Apeiros class
     *
     */
 
class Apeiros : public Serial{

public:

    /** Create Apeiros instance
    *
    * @param tx specified as PinName of UART Tx pin. Use SERIAL_TX as default. Use PA_9 for wireless shield.
    * @param rx specified as PinName of UART Rx pin. Use SERIAL_RX as deafult. Use PA_10 for wireless shield.
    * @param leftMotorPwmOffset specified as integer value from 0-150. Use 100 as default.
    * @param rightMotorPwmOffset specified as integer value from 0-150. Use 100 as default.
    * 
    * The PwmOffsets are used to eliminate the deadband for both the left & right motor, so
    * that each motor begins to rotate when motor speeds are specified as aperiros.SetMotorSpeed(1,1).
    *
    * Pass a zero for each PwmOffset parameter and then use SetMotorSpeed function to find the leftMotorPwmOffset & rightMotorPwmOffset values.
    *
    * You have found the PwmOffset value when the respective motor goes from not rotating to starting rotation.
    */
    Apeiros(PinName tx, PinName rx, int leftMotorPwmOffset, int rightMotorPwmOffset);
    
    /** Begin using Apeiros class. Must be called before using class functions.
     *
     */
    void Begin(void);
    
    /** Is Serial data available to be parsed.
     *
     * @returns
     *   1 if serial data is available,
     *   0 if no serial data has been received
     */
    bool IsSerialDataAvailable(void);
    
    /** Parse available serial UART data.
     *
     */
    void ParseUartData(void);
    
    /** Set piezo buzzer tone.
     *
     * @param buzzerTone specified as an integer value(int) from 0-100.
     */
    void SetBuzzerTone(int buzzerTone);
    
    /** Set left and right motor speeds.
     *
     * @param leftMotorSpeed specified as an integer value(int) from 0 to (255-leftMotorPwmOffset). Use a preceeding negative sign to reverse motor rotation.
     * @param rightMotorSpeed specified as an integer value(int) from 0 to (255-rightMotorPwmOffset). Use a preceeding negative sign to reverse motor rotation.
     * 
     * For example, if leftMotorPwmOffset = 95 then valid left motor speed range is 0 to 165. Use a preceeding negative sign to reverse motor rotation.
     *
     * Drive robot forward with apeiros.SetMotorSpeed(65,65).
     *
     * Drive robot in reverse with apeiros.SetMotorSpeed(-65,-65).
     *
     * Spin robot clockwise with apeiros.SetMotorSpeed(65,-65).
     *
     * Spin robot counterclockwise with apeiros.SetMotorSpeed(-65,65).
     */
    void SetMotorSpeed(int leftMotorSpeed, int rightMotorSpeed);
    
    /** Set servo gripper position.
     *
     * @param pulseWidth_us specified as an integer value(int) from MIN_GRIPPER_PULSE(1000) to MAX_GRIPPER_PULSE(2200).
     */
    void SetGripperPosition(int pulseWidth_us);
    
    /** Return integer count for left wheel encoder.
     *
     * @returns
     *   Integer value for left wheel encoder count.
     */
    int GetLeftEncoder(void);
    
    /** Return integer count for right wheel encoder.
     *
     * @returns
     *   Integer value for right wheel encoder count.
     */
    int GetRightEncoder(void);
    
    /** Reset both left & right wheel encoder counts.
     *
     */
    void ResetWheelEncoders(void);
    
    /** DigitalIn object for Left Front IR Sensor.
     *
     */
    DigitalIn leftFrontIR;
    
    /** DigitalIn object for Center Front IR Sensor.
     *
     */
    DigitalIn centerFrontIR;
    
    /** DigitalIn object for Right Front IR Sensor.
     *
     */
    DigitalIn rightFrontIR;
    
    /** DigitalIn object for Left Rear IR Sensor.
     *
     */
    DigitalIn leftRearIR;
    
    /** DigitalIn object for Right Rear IR Sensor.
     *
     */
    DigitalIn rightRearIR;
    
    /** ad_0 AnalogIn object. Used to convert a varying voltage to a digital value.
     *
     */
    AnalogIn ad_0;
    /** ad_1 AnalogIn object. Used to convert a varying voltage to a digital value.
     *
     */
    AnalogIn ad_1;
    /** ad_2 AnalogIn object. Used to convert a varying voltage to a digital value.
     *
     */
    AnalogIn ad_2;
    /** ad_3 AnalogIn object. Used to convert a varying voltage to a digital value.
     *
     */
    AnalogIn ad_3;
    /** ad_4 AnalogIn object. Used to convert a varying voltage to a digital value.
     *
     */
    AnalogIn ad_4;
    /** ad_5 AnalogIn object. Used to convert a varying voltage to a digital value.
     *
     */
    AnalogIn ad_5;
    

    
private:

    DigitalOut _buzzerOut;

    DigitalOut _SN_3A;
    DigitalOut _SN_4A;

    DigitalOut _SN_2A;
    DigitalOut _SN_1A;

    DigitalIn _leftEncoderDirPin;
    DigitalIn _rightEncoderDirPin;
    
    InterruptIn _leftEncoderClk;
    InterruptIn _rightEncoderClk;
    
    PwmOut _rightMotorPWM;
    PwmOut _leftMotorPWM;
    PwmOut _gripperPWM;
    
    Ticker _motorControl;
    
    volatile char rxBuff[16];
    volatile char rxBuffIndx;
    volatile char tmpRxBuff[16];
    volatile bool uartDataRdy;
    
    volatile int motorUpdateTickCount;
    volatile int motionLoopCount;
    
    volatile int leftEncoderCount;
    volatile int rightEncoderCount;
    volatile bool leftEncoderDir;
    volatile bool rightEncoderDir;
    volatile int leftMotorOffset;
    volatile int rightMotorOffset;
    
    volatile unsigned short analogIndex;
    volatile float adSensors[6];
    volatile int analogUpdateCount;
    
    volatile char encoderArrayIndex_L, encoderArrayIndex_R;
    volatile int encoderPeriodArray_L[4], encoderPeriodArray_R[4];
    volatile int encoderPeriod_L, encoderPeriod_R, encoderPeriodSum_L, encoderPeriodSum_R;
    volatile int prevT3Count_L, prevT3Count_R;
    volatile bool encoderUpdated_L, encoderUpdated_R;
    volatile int encoderSpeed_L, encoderSpeed_R;
    
    // Buzzer Variables //
    volatile int buzzerPeriod;
    volatile int buzzerDuty;
    volatile int buzzerTick;
    volatile bool enableBuzzer;

    void getUartData(void);
    void motorControlISR(void);
    void leftEncoderTick(void);
    void rightEncoderTick(void);
    
    void ParseMotorCommand(void);
    void ParseBuzzerCommand(void);
    void ParseGripperCommand(void);
    void CalculateWheelSpeed(void);
    void Initialize(void);

};
 
#endif