//****************************************************************************/
// Description:
//
//  Program AVR chips with the AVR910 ISP (in-system programming) protocol,
//  using an mbed.
//
// AVR910 Application Note:
//
//  http://www.atmel.com/dyn/resources/prod_documents/doc0943.pdf
//****************************************************************************/

#ifndef MBED_AVR910_H
#define MBED_AVR910_H

//****************************************************************************/
// Includes
//****************************************************************************/
#include "mbed.h"

//****************************************************************************/
// Defines
//****************************************************************************/

#define PATH_TO_BINARY "/local/AVRcode.bin"

//Commands
#define ATMEL_VENDOR_CODE 0x1E
#define DEVICE_LOCKED     0x00
#define WRITE_HIGH_BYTE   0x48
#define WRITE_LOW_BYTE    0x40
#define READ_HIGH_BYTE    0x28
#define READ_LOW_BYTE     0x20

#define PAGE_SIZE         ATMEGA328P_PAGESIZE
#define NUM_PAGES         ATMEGA328P_NUM_PAGES

//ATMega328P
#define ATMEGA328P_PAGESIZE  64 //Size in words.
#define ATMEGA328P_NUM_PAGES 256


class AVR910 {

public:

    /**
     * Constructor.
     *
     * Parameters:
     *
     *  mosi - Master out slave in pin for SPI communication.
     *  miso - Master in slave out pin for SPI communication.
     *  sclk - Serial clock pin for SPI communication.
     *  nReset - Not reset line pin on the ISP interface.
     */
    AVR910(PinName mosi, PinName miso, PinName sclk, PinName nReset);
    
    /**
     * Program the AVR microcontroller connected to the mbed.
     *
     * Parameters:
     *
     *  binary - File pointer to the binary file to be loaded onto the
     *           AVR microcontroller.
     *
     * Returns:
     *
     *   0 -> AVR microcontroller programmed successfully.
     *  -1 -> Problem during programming.
     */
    int program(FILE* binary);
    
    /**
     * Set the frequency of the SPI communication.
     *
     *  (Wrapper for SPI::frequency)
     *
     * Parameters:
     *
     *  frequency - Frequency of the SPI communication in hertz.
     */
    void setFrequency(int frequency);

private:

    /**
     * Issue an enable programming command to the AVR microcontroller.
     *
     * Returns:
     *
     *   0 to indicate programming was enabled successfully.
     *  -1 to indicate programming was not enabled.
     */
    int enableProgramming(void);
    
    /**
     * Poll the device until it has finished its current operation.
     */
    void poll(void);
    
    /**
     * Read the vendor code of the device.
     *
     * Returns:
     *
     *  The vendor code - should be 0x1E for Atmel.
     *  0x00 -> Device is locked.
     */
    int readVendorCode(void);
    
    /**
     * Read the part family and flash size of the device.
     *
     * Returns:
     *
     *  Code indicating the family of AVR microcontrollers the device comes
     *  from and how much flash memory it contains.
     *  0xFF -> Device code erased or target missing.
     *  0x01 -> Device is locked.
     */
    int readPartFamilyAndFlashSize(void);
    
    /**
     * Read the part number.
     *
     * Returns:
     *
     *  Code identifying the part number.
     *  0xFF -> Device code erased or target missing.
     *  0x02 -> Device is locked.
     */
    int readPartNumber(void);
    
    /**
     * Issue a chip erase command to the AVR microcontroller.
     */
    void chipErase(void);
    
    /**
     * Load a byte into the memory page buffer.
     *
     * Parameters:
     *
     *  highLow - Indicate whether the byte being loaded is a high or low byte.
     *  data    - The data byte to load.
     */
    void loadMemoryPage(int highLow, char address, char data);
    
    /**
     * Write the memory page buffer to flash memory.
     *
     * Parameters:
     *
     *  pageNumber - The page number to write to in flash memory.
     */
    void writeFlashMemoryPage(char pageNumber);
    
    /**
     * Read a byte from program memory.
     *
     * Parameters:
     *
     *  highLow - Indicate whether the byte being read is a low or high byte.
     *  pageNumber - The page number to read from.
     *  pageOffset - Address of byte in the page.
     *
     * Returns:
     *
     *  The byte at the specified memory location.
     */
    char readProgramMemory(int highLow, char pageNumber, char pageOffset);
    
    /**
     * Check the binary has been written correctly.
     *
     * Paramters:
     *
     *  numPages - The number of pages written to the AVR microcontroller.
     *  binary - File pointer to the binary used.
     * Returns:
     *
     *   0 -> No inconsistencies between binary file and AVR flash memory.
     *  -1 -> Binary file was not written correctly.
     */
    int checkMemory(int numPages, FILE* binary);

    DigitalOut* nReset_;
    SPI*        spi_;

};

#endif /* AVR910_H */
