/**
 * @author Aaron Berk
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * SCA3000, triple axis, digital interface, accelerometer.
 *
 * Datasheet:
 *
 * http://www.sparkfun.com/datasheets/Sensors/Accelerometer/SCA3000-D01.pdf
 */

#ifndef MBED_SCA3000_H
#define MBED_SCA3000_H

/**
 * Includes
 */
#include "mbed.h"

/**
 * Defines
 */
//Registers.
#define SCA3000_REVID_REG  0x00
#define SCA3000_STATUS_REG 0x02
#define SCA3000_X_LSB      0x04
#define SCA3000_X_MSB      0x05
#define SCA3000_Y_LSB      0x06
#define SCA3000_Y_MSB      0x07
#define SCA3000_Z_LSB      0x08
#define SCA3000_Z_MSB      0x09
 
#define SCA3000_SPI_READ   0x00
#define SCA3000_SPI_WRITE  0x02

#define SCA3000_X_AXIS     0x00
#define SCA3000_Y_AXIS     0x01
#define SCA3000_Z_AXIS     0x02

/**
 * SCA3000, triple axis, digital interface, accelerometer.
 */
class SCA3000 {

public:

    /**
     * Constructor.
     *
     * @param mosi mbed pin to use for MOSI line of SPI interface.
     * @param miso mbed pin to use for MISO line of SPI interface.
     * @param sck mbed pin to use for SCK line of SPI interface.
     * @param cs mbed pin to use for not chip select line of SPI interface.
     * @param nr mbed pin to use for the not reset line.
     */
    SCA3000(PinName mosi, PinName miso, PinName sck, PinName cs, PinName nr);
    
    /**
     * Read the revision ID register on the device.
     *
     * @return The revision ID number.
     */
    int getRevId(void);
    
    /**
     * Get the register contents acceleration value for the
     * given axis, using the nominal sensitivity values
     * found in the datasheet.
     *
     * @param axis The axis to get acceleration values for.
     *
     * @return The acceleration on the specified axis in mg.
     */
    float getAcceleration(int axis);
    
    /**
     * Get the register contents acceleration value for the
     * given axis, in counts.
     *
     * @param axis The axis to get the counts values for.
     *
     * @return The acceleration on the specified axis in counts.
     */
    int getCounts(int axis);

private:

    SPI        spi_;
    DigitalOut nCS_;
    DigitalOut nR_;
    
    /**
     * Read one byte from a register on the device.
     *
     * @param address Address of the register to read.
     *
     * @return The contents of the register address.
     */
    int oneByteRead(int address);
    
    /**
     * Write one byte to a register on the device.
     *
     * @param address Address of the register to write to.
     * @param data The data to write into the register.
     */
    void oneByteWrite(int address, char data);

};

#endif /* MBED_SCA3000_H */
