/**
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * AM-MC1 Brushed Motor application module for an RS-EDP base board.
 *
 * There are a large number of options to choose from on the MC1 module.
 * After creating an RSEDP_AM_MC1 object, select which pins you wish to use
 * by making calls to "setXXXPin(pYY)" with appropriate pin you want to use.
 *
 * The pins available and their associated jumpers which need to be set
 * are detailed with the function declarations.
 *
 * See the datasheet for more information about configuring the module for
 * your application's needs.
 *
 * Datasheet:
 *
 * http://www.hitex.com/fileadmin/uk-files/EDP/Modules/MC1/MC1%20datasheet.pdf
 */

#ifndef RSEDP_AM_MC1_H
#define RSEDP_AM_MC1_H

/**
 * Includes
 */
#include "mbed.h"

/**
 * Defines
 */
#define FORWARD   1
#define BACKWARD  0
#define BRAKE_ON  1
#define BRAKE_OFF 0

/**
 * RS-EDP Brushed Motor Control Application Module library.
 */
class RSEDP_AM_MC1 {

public:

    /**
     * Constructor.
     */
    RSEDP_AM_MC1();

    /**
     * Initialize a motor control module object.
     *
     * Set PWM frequency to 20kHz, brake on, direction "forward", speed 0.
     */
    void initialize(void);

    /**
     * Set the PWM pin.
     *
     * p23 is default for PWM.
     *
     * In order to use p27 as PWM, it must be linked to EVG4_GPIO48 on the
     * command module. Default track connection is to ASC0_RX_TTL.
     *
     * @param pwm p23 [JP204 1-2] XOR p27 [JP204 2-3].
     *            Required: JP203 2-3, JP205 1-2.
     */
    void setPwmPin(PinName pwm);

    /**
     * Set the maximum current in motor windings pin.
     *
     * p21 is the default for setting the maximum current in the motor
     * windings for a simple fixed-on time mode. Chopping frequency is ~25KHz.
     * It is recommend to use PWM pin 23 instead of this.
     *
     * In order to use p18 as setting the maximum current in the motor
     * windings, it must be linked to CPU_DACO0_GPIO17 on the command module.
     * The default track connection is to AN12.
     *
     * In order to use p22 as setting the maximum current in the motor
     * windings, it must be linked to CPU_DACO1_GPIO19. This is the default
     * track connection.
     *
     * @param maxCurrent p18, p21 [JP202 1-2] XOR p22 [JP202 2-3].
     */
    void setMaxCurrentPin(PinName maxCurrent);

    /**
     * Set the motor direction pin.
     *
     * p24 is default for Motor Direction.
     *
     * In order to use p29 as Motor Direction, it must be linked to
     * EVG5_GPIO50 on the command module. This is the default track
     * connection.
     *
     * @param motorDir p24 [JP206 1-2] XOR p29 [JP206 2-3].
     *                 Required: JP207 2-3.
     */
    void setDirectionPin(PinName direction);

    /**
     * Set the motor brake pin.
     *
     * In order to use p19 as Motor Brake, it must be linked to GPIO0 on the
     * mbed command module. The default track connection is to AN4.
     *
     * @param motorBrake p19 [JP209 2-3].
     */
    void setBrakePin(PinName brake);

    /**
     * Set the VDCLINK pin.
     *
     * p17 is default for VDCLINK.
     *
     * In order to use p18 as VDCLINK, it must be linked to AN12 on the
     * command module. This is the default track connection.
     *
     * @param vdclink p17 [JP312 1-2] XOR p18 [JP312 2-3].
     */
    void setVdclinkPin(PinName vdclink);

    /**
     * Set the VSENSE pin.
     *
     * p15 is default for VSENSE.
     *
     * In order to use p16 as VSENSE, it must be linked to AN10 on the
     * command module. This is the default track connection.
     *
     * @param vsense p15 [JP307 1-2] XOR p16 [JP307 2-3].
     */
    void setVsensePin(PinName vsense);

    /**
     * Set the VTACH pin.
     *
     * In order to use p19 as VTACH, it must be linked to AN4 on the
     * command module. This is the default track connection.
     *
     * @param vtach p19 [JP304 1-2].
     */
    void setVtachPin(PinName vtach);

    /**
     * Set the open limit pin.
     *
     * In order to use p11 as Open Limit Switch, it must be linked to
     * EVM2_GPIO41_CAPADC on the command module. The default track connection
     * is to GPIO12_MCICMD.
     *
     * @param openLimit p11 [JP314 1-2].
     */
    void setOpenLimitPin(PinName openLimit);

    /**
     * Set the closed limit pin.
     *
     * In order to use p7 as Closed Limit Switch, it must be linked to
     * EVG8_GPIO56 on the command module. The default track connection is to
     * CNTRL_SPICLK.
     *
     * @param closedLimit p7 [JP309 2-3].
     */
    void setClosedLimitPin(PinName closedLimit);

    /**
     * Set the tachogenerator pulses pin.
     *
     * In order to use p30 as Tachogenerator Pulses, it must be linked to
     * EVG6_GPIO52 on the command module. The is the default track connection.
     *
     * @param tachoPulses p30 [JP302 2-3].
     *                    Required: JP303 1-2.
     */
    void setTachoPulsesPin(PinName tachoPulses);

    /**
     * Set the encoder input 0 pin.
     *
     * In order to use p30 as Quadrature Encoder 0/Hall Sensor 0, it must be
     * linked to EVG6_GPIO52 on the command module. This is the default track
     * connection.
     *
     * @param encoder0 p30 [JP302 2-3].
     *                 Required: JP303 2-3.
     */
    void setEncoder0Pin(PinName encoder0);

    /**
     * Set the encoder input 1 pin.
     *
     * In order to use p8 as Quadrature Encoder 1/Hall Sensor 1, it must be
     * linked to EVG7_GPIO54 on the command module. The default track
     * connection is to #CS_LCD.
     *
     * @param encoder1 p8 [JP306 2-3].
     */
    void setEncoder1Pin(PinName encoder1);

    /**
     * Set the motor run/stop pin.
     *
     * In order to use p26 as Motor Run/Stop, it must be linked to EVG3_GPIO46
     * on the command module. The default track connection is to ASC1_TX_TTL.
     *
     * @param runStop p26 [JP311 1-2].
     */
    void setRunStopPin(PinName runStop);

    /**
     * Get the current PWM duty cycle.
     *
     * @return A floating-point value representing the current duty-cycle being
     *         output on the pin, measured as a percentage. The returned value
     *         will lie between 0.0f (0%) and 1.0f (100%).
     */
    float getPwmDuty(void);

    /**
     * Set the PWM duty cycle.
     *
     * @param duty A floating-point value representing the output duty-cycle,
     *             specified as a percentage. The value should lie between
     *             0.0f (representing on 0%) and 1.0f (representing on 100%).
     *             Values outside this range will be saturated to 0.0f or 1.0f.
     */
    void setPwmDuty(float duty);

    /**
     * Get the maximum current in motor windings duty cycle.
     *
     * @return A floating-point value representing the current duty-cycle being
     *         output on the pin, measured as a percentage. The returned value
     *         will lie between 0.0f (0%) and 1.0f (100%).
     */
    float getMaxCurrentDuty(void);

    /**
     * Set the maximum current in the motor windings duty cycle.
     *
     * @param duty A floating-point value representing the output duty-cycle,
     *             specified as a percentage. The value should lie between
     *             0.0f (0%) and 1.0f (100%). Values outside this range will
     *             be saturated to 0.0f or 1.0f.
     */
    void setMaxCurrentDuty(float duty);

    /**
     * Set the direction of the motor.
     *
     * Sets the motor direction to "Forward" (1, clockwise)
     * or "Backward" (0, counter-clockwise).
     *
     * @param direction 1 -> "Forward"/Clockwise
     *                  0 -> "Backward"/Counter-clockwise
     */
    void setDirection(int direction);
    
    /**
     * Read the direction of the motor.
     *
     * @return The direction of the motor.
     *         1 -> "Forward"/Clockwise
     *         0 -> "Backward"/Counter-clockwise
     */
    int readDirection(void);

    /**
     * Set whether the brake is on or off.
     *
     * @param brakeSetting 1 -> Brake on.
     *                     0 -> Brake off.
     */
    void setBrake(int brakeSetting);

    /**
     * Read the motor drive voltage.
     *
     * @return The motor drive voltage in millivolts.
     */
    int readVdclink(void);

    /**
     * Read the motor current as voltage.
     *
     * @return The motor current as a voltage in millivolts.
     */
    int readVsense(void);

    /**
     * Read tachogenerator output.
     *
     * @return Output from a tachogenerator in millivolts.
     */
    int readVtach(void);

    /**
     * Read the open limit switch input.
     *
     * @return The value of the open limit switch input {0,1}.
     */
    int readOpenLimit(void);

    /**
     * Read the closed limit switch input.
     *
     * @return The value of the closed limit switch input {0,1}.
     */
    int readClosedLimit(void);

    /**
     * Read pulse input from a tachogenerator.
     *
     * @return The value of the tacho pulse input {0,1}.
     */
    int readTachoPulses(void);

    /**
     * Read channel 0 from a quadrature encoder/hall sensor.
     *
     * @return The value on channel 0 of the sensor {0,1}.
     */
    int readEncoder0(void);

    /**
     * Read channel 1 from a quadrature encoder/hall sensor.
     *
     * @return value on channel 1 of the sensor {0,1}.
     */
    int readEncoder1(void);

    /**
     * Read motor run/stop input.
     *
     * @return value on the run/stop pin {0,1}.
     */
    int readRunStop(void);

private:

    PwmOut*     pwm_;
    PwmOut*     maxCurrent_;
    DigitalOut* dir_;
    DigitalOut* brake_;
    AnalogIn*   vdclink_;
    AnalogIn*   vsense_;
    AnalogIn*   vtach_;
    DigitalIn*  openLimit_;
    DigitalIn*  closedLimit_;
    DigitalIn*  tachoPulses_;
    DigitalIn*  encoder0_;
    DigitalIn*  encoder1_;
    DigitalIn*  runStop_;

};

#endif /* RSEDP_AM_MC1_H */
