/**
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * AM-MC1 Brushed Motor application module for an RS-EDP base board.
 *
 * There are a large number of options to choose from on the MC1 module.
 * After creating an RSEDP_AM_MC1 object, select which pins you wish to use
 * by making calls to "setXXXPin(pYY)" with appropriate pin you want to use.
 *
 * The pins available and their associated jumpers which need to be set
 * are detailed with the function declarations.
 *
 * See the datasheet for more information about configuring the module for
 * your application's needs.
 *
 * Datasheet:
 *
 * http://www.hitex.com/fileadmin/uk-files/EDP/Modules/MC1/MC1%20datasheet.pdf
 */

/**
 * Includes
 */
#include "RSEDP_AM_MC1.h"

RSEDP_AM_MC1::RSEDP_AM_MC1() {

}

void RSEDP_AM_MC1::initialize(void) {

    //20KHz.
    pwm_->period_us(50);
    //Off.
    *pwm_ = 1.0;

    *brake_ = BRAKE_ON;

    *dir_ = FORWARD;

}

void RSEDP_AM_MC1::setPwmPin(PinName pwm) {

    pwm_ = new PwmOut(pwm);

}

void RSEDP_AM_MC1::setMaxCurrentPin(PinName maxCurrent) {

    maxCurrent_ = new PwmOut(maxCurrent);

}

void RSEDP_AM_MC1::setDirectionPin(PinName dir) {

    dir_ = new DigitalOut(dir);

}

void RSEDP_AM_MC1::setBrakePin(PinName brake) {

    brake_ = new DigitalOut(brake);

}

void RSEDP_AM_MC1::setVdclinkPin(PinName vdclink) {

    vdclink_ = new AnalogIn(vdclink);

}

void RSEDP_AM_MC1::setVsensePin(PinName vsense) {

    vsense_ = new AnalogIn(vsense);

}

void RSEDP_AM_MC1::setVtachPin(PinName vtach) {

    vtach_ = new AnalogIn(vtach);

}

void RSEDP_AM_MC1::setOpenLimitPin(PinName openLimit) {

    openLimit_ = new DigitalIn(openLimit);

}

void RSEDP_AM_MC1::setClosedLimitPin(PinName closedLimit) {

    closedLimit_ = new DigitalIn(closedLimit);

}

void RSEDP_AM_MC1::setTachoPulsesPin(PinName tachoPulses) {

    tachoPulses_ = new DigitalIn(tachoPulses);

}

void RSEDP_AM_MC1::setEncoder0Pin(PinName encoder0) {

    encoder0_ = new DigitalIn(encoder0);

}

void RSEDP_AM_MC1::setEncoder1Pin(PinName encoder1) {

    encoder1_ = new DigitalIn(encoder1);

}

void RSEDP_AM_MC1::setRunStopPin(PinName runStop) {

    runStop_ = new DigitalIn(runStop);

}

float RSEDP_AM_MC1::getPwmDuty(void) {

    return pwm_->read();

}

void RSEDP_AM_MC1::setPwmDuty(float duty) {

    pwm_->write(duty);

}

float RSEDP_AM_MC1::getMaxCurrentDuty(void) {

    return maxCurrent_->read();

}

void RSEDP_AM_MC1::setDirection(int direction) {

    //Check for valid direction.
    if (direction == FORWARD) {
        *dir_ = FORWARD;
    } else if (direction == BACKWARD) {
        *dir_ = BACKWARD;
    }

}

int RSEDP_AM_MC1::readDirection(void){

    return dir_->read();

}

void RSEDP_AM_MC1::setBrake(int brakeSetting) {

    //Check for valid setting.
    if (brakeSetting == BRAKE_ON) {
        *brake_ = BRAKE_ON;
    } else if (brakeSetting == BRAKE_OFF) {
        *brake_ = BRAKE_OFF;
    }

}

int RSEDP_AM_MC1::readVdclink(void) {

    return (vdclink_->read())*3300;

}

int RSEDP_AM_MC1::readVsense(void) {

    return (vsense_->read())*3300;

}

int RSEDP_AM_MC1::readVtach(void) {

    return (vtach_->read())*3300;

}

int RSEDP_AM_MC1::readOpenLimit(void) {

    return openLimit_->read();

}

int RSEDP_AM_MC1::readClosedLimit(void) {

    return closedLimit_->read();

}

int RSEDP_AM_MC1::readTachoPulses(void) {

    return tachoPulses_->read();

}

int RSEDP_AM_MC1::readEncoder0(void) {

    return encoder0_->read();

}

int RSEDP_AM_MC1::readEncoder1(void) {

    return encoder1_->read();

}

int RSEDP_AM_MC1::readRunStop(void) {

    return runStop_->read();

}
